# 🎨 توثيق نظام الـ Frontend

## 📋 نظرة عامة

تم إنشاء نظام **Frontend** منفصل تماماً عن النظام الداخلي للطلاب، مخصص لعرض محتوى تسويقي للزوار والمهتمين.

---

## 🗂️ البنية التقنية

### **الجداول (Database Tables):**

#### 1. `frontend_course_categories`
```sql
- id
- name                  (اسم التصنيف)
- slug                  (الرابط الفريد)
- description           (الوصف)
- icon                  (أيقونة)
- image                 (صورة)
- color                 (لون مميز)
- is_active             (فعّال؟)
- order                 (الترتيب)
- created_at, updated_at
```

#### 2. `frontend_courses`
```sql
- id
- title, slug, subtitle
- description
- category_id           (FK → frontend_course_categories)
- instructor_id         (FK → users)
- what_you_learn        (JSON - ماذا سيتعلم الطالب)
- requirements          (المتطلبات)
- level                 (beginner, intermediate, advanced)
- language              (ar, en)
- duration              (ساعات)
- lessons_count         (عدد الدروس)
- thumbnail, preview_video, cover_image
- price, discount_price, is_free, currency
- status                (draft, published, archived)
- is_featured           (مميز؟)
- is_active             (فعّال؟)
- published_at
- students_count, rating, reviews_count, views_count
- meta_title, meta_description, meta_keywords
- certificate, lifetime_access, downloadable_resources
- order
- created_at, updated_at, deleted_at
```

#### 3. `frontend_reviews`
```sql
- id
- frontend_course_id    (FK → frontend_courses - nullable)
- student_name
- student_email
- student_image
- student_position      (مثل: "مطور ويب", "طالب")
- rating                (1-5 نجوم)
- review_text
- is_active
- is_featured
- order
- created_at, updated_at
```

---

## 📦 الـ Models

### **1. FrontendCourse**
**المسار:** `app/Models/FrontendCourse.php`

**العلاقات:**
- `category()` → `FrontendCourseCategory`
- `instructor()` → `User`
- `reviews()` → `FrontendReview`

**Scopes:**
- `active()` - الكورسات الفعّالة
- `published()` - الكورسات المنشورة
- `featured()` - الكورسات المميزة
- `free()` - الكورسات المجانية
- `paid()` - الكورسات المدفوعة
- `byLevel($level)` - حسب المستوى
- `byCategory($categoryId)` - حسب التصنيف

**Helper Methods:**
- `getEffectivePriceAttribute()` - السعر الفعلي (بعد الخصم)
- `hasDiscount()` - هل يوجد خصم؟
- `getDiscountPercentageAttribute()` - نسبة الخصم
- `incrementViews()` - زيادة عدد المشاهدات
- `updateRating()` - تحديث التقييم
- `getLevelNameAttribute()` - اسم المستوى بالعربية
- `getStatusNameAttribute()` - اسم الحالة بالعربية

---

### **2. FrontendCourseCategory**
**المسار:** `app/Models/FrontendCourseCategory.php`

**العلاقات:**
- `courses()` → `FrontendCourse`

**Scopes:**
- `active()` - التصنيفات الفعّالة

**Helper Methods:**
- `getCoursesCountAttribute()` - عدد الكورسات في التصنيف

---

### **3. FrontendReview**
**المسار:** `app/Models/FrontendReview.php`

**العلاقات:**
- `course()` → `FrontendCourse` (nullable)

**Scopes:**
- `active()` - المراجعات الفعّالة
- `featured()` - المراجعات المميزة
- `byRating($rating)` - حسب التقييم

**Helper Methods:**
- `getStarsAttribute()` - نجوم Emoji (⭐⭐⭐⭐⭐)
- `getStarsHtmlAttribute()` - نجوم HTML بـ Font Awesome

---

## 🎮 الـ Controllers

### **1. HomeController**
**المسار:** `app/Http/Controllers/Frontend/HomeController.php`

**Methods:**
- `index()` - الصفحة الرئيسية (6 مراجعات + 8 كورسات مميزة)
- `reviews()` - صفحة جميع المراجعات (مع Pagination)
- `contact()` - صفحة اتصل بنا
- `sendContact()` - إرسال رسالة تواصل

---

### **2. CourseController**
**المسار:** `app/Http/Controllers/Frontend/CourseController.php`

**Methods:**
- `index(Request $request)` - صفحة الكورسات مع فلاتر
  - Filter by category
  - Filter by level
  - Filter by price (free/paid)
  - Sort by: latest, popular, rating, price_low, price_high

- `show($slug)` - تفاصيل كورس واحد
  - عرض الكورس
  - زيادة عدد المشاهدات
  - كورسات مشابهة (4 من نفس التصنيف)

---

## 🛣️ الـ Routes

**الملف:** `routes/frontend.php`

```php
GET  /                      → home (الصفحة الرئيسية)
GET  /courses               → courses.index (قائمة الكورسات)
GET  /courses/{slug}        → courses.show (تفاصيل كورس)
GET  /reviews               → reviews (المراجعات)
GET  /contact               → contact (اتصل بنا)
POST /contact               → contact.send (إرسال رسالة)
```

**ملاحظة:** صفحة التحقق من الشهادات موجودة في:
```
GET /certificates/verify    (مشتركة بين Frontend و Backend)
```

---

## 🎨 الـ Views

### **البنية:**
```
resources/views/frontend/
├── layouts/
│   ├── master.blade.php        (الـ Layout الرئيسي)
│   ├── head.blade.php          (الـ CSS والـ Meta)
│   ├── main-header.blade.php   (الـ Header)
│   └── footer.blade.php        (الـ Footer)
├── sections/
│   ├── hero.blade.php          (Hero Section)
│   ├── services-section.blade.php
│   ├── courses-section.blade.php
│   └── testimonials.blade.php  (المراجعات)
└── pages/
    ├── index.blade.php         (الصفحة الرئيسية)
    ├── courses.blade.php       (قائمة الكورسات)
    ├── course-details.blade.php
    ├── reviews.blade.php
    └── contact.blade.php
```

---

## 🎯 الـ Assets

**المسار:** `public/frontend/assets/`

```
frontend/assets/
├── css/
│   ├── bootstrap.css
│   └── style.css
├── images/
│   ├── global/
│   ├── courses/
│   └── services/
└── bootstrap.js
```

**استخدام الـ Assets في Blade:**
```blade
<link rel="stylesheet" href="{{ asset('frontend/assets/css/bootstrap.css') }}">
<link rel="stylesheet" href="{{ asset('frontend/assets/css/style.css') }}">
<script src="{{ asset('frontend/assets/bootstrap.js') }}"></script>
```

---

## 🚀 كيفية الاستخدام

### **1. إضافة تصنيف جديد:**

```php
use App\Models\FrontendCourseCategory;

FrontendCourseCategory::create([
    'name' => 'تطوير الويب',
    'slug' => 'web-development',
    'description' => 'كورسات تطوير المواقع والتطبيقات',
    'icon' => 'fas fa-code',
    'color' => '#667eea',
    'is_active' => true,
    'order' => 1,
]);
```

---

### **2. إضافة كورس جديد:**

```php
use App\Models\FrontendCourse;

FrontendCourse::create([
    'title' => 'تطوير تطبيقات الويب بـ Laravel',
    'slug' => 'laravel-web-development',
    'subtitle' => 'من الصفر إلى الاحتراف',
    'description' => 'كورس شامل لتعلم Laravel...',
    'category_id' => 1,
    'instructor_id' => 1,
    'what_you_learn' => [
        'أساسيات Laravel',
        'بناء API',
        'قواعد البيانات',
        'Authentication',
    ],
    'requirements' => 'معرفة أساسية بـ PHP',
    'level' => 'intermediate',
    'language' => 'ar',
    'duration' => 40.5,
    'lessons_count' => 120,
    'thumbnail' => 'courses/laravel.jpg',
    'price' => 499.00,
    'discount_price' => 399.00,
    'is_free' => false,
    'currency' => 'SAR',
    'status' => 'published',
    'is_featured' => true,
    'is_active' => true,
    'published_at' => now(),
    'certificate' => true,
    'meta_title' => 'كورس Laravel - تطوير الويب',
    'meta_description' => 'أفضل كورس لتعلم Laravel...',
]);
```

---

### **3. إضافة مراجعة:**

```php
use App\Models\FrontendReview;

FrontendReview::create([
    'frontend_course_id' => 1, // أو null للمراجعات العامة
    'student_name' => 'أحمد محمد',
    'student_email' => 'ahmed@example.com',
    'student_image' => 'students/ahmed.jpg',
    'student_position' => 'مطور Full Stack',
    'rating' => 5,
    'review_text' => 'كورس ممتاز جداً! استفدت كثيراً...',
    'is_active' => true,
    'is_featured' => true,
    'order' => 1,
]);
```

---

## 📊 عرض البيانات في Views

### **في الصفحة الرئيسية:**

```blade
@foreach($courses as $course)
    <div class="course-card">
        <img src="{{ asset('storage/' . $course->thumbnail) }}">
        <h3>{{ $course->title }}</h3>
        <p>{{ $course->subtitle }}</p>

        @if($course->hasDiscount())
            <span class="old-price">{{ $course->price }} {{ $course->currency }}</span>
            <span class="discount">-{{ $course->discount_percentage }}%</span>
        @endif

        <span class="price">{{ $course->effective_price }} {{ $course->currency }}</span>

        <div class="rating">
            {{ str_repeat('⭐', round($course->rating)) }}
            ({{ $course->reviews_count }})
        </div>

        <span class="level">{{ $course->level_name }}</span>
        <span class="duration">{{ $course->duration }} ساعة</span>
    </div>
@endforeach
```

---

### **عرض المراجعات:**

```blade
@foreach($reviews as $review)
    <div class="review-card">
        <img src="{{ asset('storage/' . $review->student_image) }}">
        <h4>{{ $review->student_name }}</h4>
        <p class="position">{{ $review->student_position }}</p>

        <div class="stars">
            {!! $review->stars_html !!}
        </div>

        <p class="review-text">{{ $review->review_text }}</p>
    </div>
@endforeach
```

---

## 🔍 الفلاتر والبحث

### **مثال: صفحة الكورسات مع فلاتر:**

```php
// في Controller
$query = FrontendCourse::with(['category', 'instructor'])
              ->active()
              ->published();

// Filter by category
if ($request->has('category')) {
    $query->where('category_id', $request->category);
}

// Filter by level
if ($request->has('level')) {
    $query->where('level', $request->level);
}

// Filter by price
if ($request->price_type == 'free') {
    $query->where('is_free', true);
} elseif ($request->price_type == 'paid') {
    $query->where('is_free', false);
}

// Sort
switch ($request->get('sort', 'latest')) {
    case 'popular':
        $query->orderBy('students_count', 'desc');
        break;
    case 'rating':
        $query->orderBy('rating', 'desc');
        break;
    case 'price_low':
        $query->orderBy('price', 'asc');
        break;
    case 'price_high':
        $query->orderBy('price', 'desc');
        break;
    default:
        $query->orderBy('created_at', 'desc');
}

$courses = $query->paginate(12);
```

---

## ✅ ملخص الميزات

### **الكورسات:**
- ✅ تصنيفات منفصلة
- ✅ معلومات كاملة (سعر، خصم، مستوى، مدة)
- ✅ صور ومقاطع معاينة
- ✅ تقييمات ومراجعات
- ✅ SEO Friendly (Meta Tags)
- ✅ فلاتر متقدمة
- ✅ Soft Deletes

### **المراجعات:**
- ✅ مراجعات عامة أو مرتبطة بكورس
- ✅ تقييمات 1-5 نجوم
- ✅ صور وبيانات الطالب
- ✅ تحكم في النشر (is_active, is_featured)

### **التصنيفات:**
- ✅ أيقونات وألوان مخصصة
- ✅ ترتيب قابل للتخصيص
- ✅ عدّاد الكورسات تلقائي

---

## 🔗 الربط مع نظام الشهادات

**صفحة التحقق من الشهادات موجودة بالفعل:**
- المسار: `/certificates/verify`
- يمكن الوصول لها من Frontend و Backend
- لا داعي لإنشاء صفحة منفصلة

**للربط في Footer:**
```blade
<a href="{{ route('certificates.verify.index') }}">
    التحقق من الشهادات
</a>
```

---

## 📝 ملاحظات مهمة

### **الفرق بين النظامين:**

| Frontend System | Internal System |
|----------------|-----------------|
| `frontend_courses` | `courses` (النظام الداخلي) |
| `frontend_reviews` | `course_reviews` |
| `frontend_course_categories` | تصنيفات النظام الداخلي |
| للزوار والتسويق | للطلاب المسجلين |
| عرض فقط | تفاعل كامل (دروس، واجبات، إلخ) |

---

## 🎉 البيانات التجريبية (Seed Data)

تم إضافة بيانات تجريبية كاملة للاختبار:

### **إحصائيات البيانات:**
- ✅ **8 تصنيفات** (تطوير الويب، الجوال، Python، التصميم، تحليل البيانات، التسويق، AI، الأمن السيبراني)
- ✅ **13 كورس** موزعة على جميع التصنيفات
  - 8 كورسات مميزة (Featured)
  - 5 كورسات مجانية
  - 8 كورسات مدفوعة مع خصومات
- ✅ **16 مراجعة** (6 عامة + 10 مرتبطة بكورسات محددة)

### **تشغيل السيدرز:**
```bash
php artisan db:seed --class=FrontendCategoriesSeeder
php artisan db:seed --class=FrontendCoursesSeeder
php artisan db:seed --class=FrontendReviewsSeeder
```

### **أمثلة على الكورسات المضافة:**
1. تطوير تطبيقات الويب بـ Laravel (499 SAR - خصم 20%)
2. بناء واجهات تفاعلية بـ Vue.js (399 SAR - خصم 25%)
3. تطوير مواقع ووردبريس احترافية (مجاني)
4. تطوير تطبيقات Android بـ Kotlin (599 SAR - خصم 25%)
5. Flutter للمبتدئين (مجاني)
6. Python من الصفر للاحتراف (449 SAR - خصم 22%)
... وغيرها

### **الوصول للصفحات:**
- الصفحة الرئيسية: `http://localhost:8000/`
- قائمة الكورسات: `http://localhost:8000/courses`
- تفاصيل كورس: `http://localhost:8000/courses/laravel-web-development`
- المراجعات: `http://localhost:8000/reviews`
- اتصل بنا: `http://localhost:8000/contact`

---

**تاريخ الإنشاء:** 2025-11-29
**الإصدار:** 1.0
**الحالة:** ✅ جاهز للاستخدام والاختبار
