# 🎓 المخطط الكامل لنظام إدارة التعلم (LMS)
## Moodle-like Learning Management System - Blueprint

**آخر تحديث:** 2025-11-12
**الإصدار:** 1.0
**المطور:** كلاودسوفت

---

## 📋 جدول المحتويات

1. [نظرة عامة](#نظرة-عامة)
2. [البنية الهرمية](#البنية-الهرمية)
3. [قاعدة البيانات الكاملة](#قاعدة-البيانات-الكاملة)
4. [الأنظمة الفرعية](#الأنظمة-الفرعية)
5. [نظام الاختبارات](#نظام-الاختبارات)
6. [نظام الاختبارات البرمجية](#نظام-الاختبارات-البرمجية)
7. [نظام الفواتير والمدفوعات](#نظام-الفواتير-والمدفوعات)
8. [التقنيات المستخدمة](#التقنيات-المستخدمة)
9. [خطة التنفيذ](#خطة-التنفيذ)

---

## 🎯 نظرة عامة

نظام إدارة تعلم متكامل مشابه لـ Moodle يدعم:
- ✅ الكورسات والدروس التفاعلية
- ✅ الاختبارات بجميع أنواعها (12 نوع)
- ✅ الاختبارات البرمجية مع محرر أكواد
- ✅ الواجبات والمشاريع
- ✅ المعسكرات التدريبية
- ✅ نظام الفواتير والمدفوعات
- ✅ الشهادات والتقييمات
- ✅ المنتديات والنقاشات
- ✅ التقارير والتحليلات

---

## 🏗️ البنية الهرمية

```
Course Categories (التصنيفات)
    ↓
Courses (الكورسات)
    ↓
Course Sections (الأقسام/الوحدات)
    ↓
Course Modules (الوحدات التعليمية)
    ├─ Lessons (دروس نصية)
    ├─ Videos (فيديوهات)
    ├─ Resources (ملفات)
    ├─ Quizzes (اختبارات)
    ├─ Programming Challenges (اختبارات برمجية)
    ├─ Assignments (واجبات)
    ├─ Discussions (منتديات)
    └─ Live Sessions (جلسات مباشرة)
```

---

## 🗄️ قاعدة البيانات الكاملة

### المجموعة 1: البنية التحتية للكورسات (8 جداول)

#### 1. `courses`
```sql
CREATE TABLE courses (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    course_category_id BIGINT UNSIGNED NOT NULL,
    title VARCHAR(255) NOT NULL,
    slug VARCHAR(255) UNIQUE NOT NULL,
    code VARCHAR(50) UNIQUE,
    description LONGTEXT,
    short_description TEXT,
    image VARCHAR(255),
    instructor_id BIGINT UNSIGNED,
    level ENUM('beginner', 'intermediate', 'advanced', 'expert') DEFAULT 'beginner',
    language ENUM('ar', 'en', 'both') DEFAULT 'ar',
    duration_in_hours INT,
    price DECIMAL(10,2) DEFAULT 0,
    is_free BOOLEAN DEFAULT FALSE,
    is_published BOOLEAN DEFAULT FALSE,
    enrollment_type ENUM('open', 'by_approval', 'invite_only') DEFAULT 'open',
    max_students INT NULL,
    start_date DATE,
    end_date DATE NULL,
    completion_criteria JSON,
    sort_order INT DEFAULT 0,
    meta_keywords TEXT,
    meta_description TEXT,
    created_by BIGINT UNSIGNED,
    updated_by BIGINT UNSIGNED,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    deleted_at TIMESTAMP NULL,

    FOREIGN KEY (course_category_id) REFERENCES course_categories(id) ON DELETE RESTRICT,
    FOREIGN KEY (instructor_id) REFERENCES users(id) ON DELETE SET NULL,
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_published (is_published),
    INDEX idx_category (course_category_id),
    INDEX idx_instructor (instructor_id)
);
```

#### 2. `course_sections`
```sql
CREATE TABLE course_sections (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    course_id BIGINT UNSIGNED NOT NULL,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    summary TEXT,
    sort_order INT DEFAULT 0,
    is_visible BOOLEAN DEFAULT TRUE,
    is_locked BOOLEAN DEFAULT FALSE,
    unlock_conditions JSON,
    start_date DATE NULL,
    end_date DATE NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    deleted_at TIMESTAMP NULL,

    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE,
    INDEX idx_course (course_id),
    INDEX idx_order (sort_order)
);
```

#### 3. `course_modules`
```sql
CREATE TABLE course_modules (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    course_id BIGINT UNSIGNED NOT NULL,
    section_id BIGINT UNSIGNED,
    module_type ENUM('lesson', 'video', 'quiz', 'programming_challenge', 'assignment', 'resource', 'forum', 'live_session') NOT NULL,
    modulable_id BIGINT UNSIGNED NOT NULL,
    modulable_type VARCHAR(255) NOT NULL,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    sort_order INT DEFAULT 0,
    is_visible BOOLEAN DEFAULT TRUE,
    is_required BOOLEAN DEFAULT FALSE,
    is_graded BOOLEAN DEFAULT FALSE,
    max_score DECIMAL(10,2) NULL,
    completion_type ENUM('auto', 'manual', 'score_based') DEFAULT 'auto',
    estimated_duration INT COMMENT 'minutes',
    unlock_conditions JSON,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    deleted_at TIMESTAMP NULL,

    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE,
    FOREIGN KEY (section_id) REFERENCES course_sections(id) ON DELETE SET NULL,
    INDEX idx_type (module_type),
    INDEX idx_polymorphic (modulable_id, modulable_type)
);
```

#### 4. `course_groups`
```sql
CREATE TABLE course_groups (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    course_id BIGINT UNSIGNED NOT NULL,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    max_members INT NULL,
    image VARCHAR(255),
    is_visible BOOLEAN DEFAULT TRUE,
    created_by BIGINT UNSIGNED,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    deleted_at TIMESTAMP NULL,

    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE,
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL
);
```

#### 5. `course_group_members`
```sql
CREATE TABLE course_group_members (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    group_id BIGINT UNSIGNED NOT NULL,
    student_id BIGINT UNSIGNED NOT NULL,
    role ENUM('member', 'leader') DEFAULT 'member',
    joined_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (group_id) REFERENCES course_groups(id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_member (group_id, student_id)
);
```

#### 6. `course_instructors`
```sql
CREATE TABLE course_instructors (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    course_id BIGINT UNSIGNED NOT NULL,
    instructor_id BIGINT UNSIGNED NOT NULL,
    role ENUM('main_instructor', 'co_instructor', 'teaching_assistant') DEFAULT 'co_instructor',
    permissions JSON,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE,
    FOREIGN KEY (instructor_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_instructor (course_id, instructor_id)
);
```

#### 7. `course_enrollments`
```sql
CREATE TABLE course_enrollments (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    course_id BIGINT UNSIGNED NOT NULL,
    student_id BIGINT UNSIGNED NOT NULL,
    enrollment_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    enrollment_status ENUM('pending', 'active', 'completed', 'suspended', 'cancelled') DEFAULT 'active',
    completion_percentage DECIMAL(5,2) DEFAULT 0,
    enrolled_by BIGINT UNSIGNED NULL COMMENT 'admin_id or null for self-enrollment',
    progress JSON,
    last_accessed_at TIMESTAMP NULL,
    completed_at TIMESTAMP NULL,
    certificate_issued BOOLEAN DEFAULT FALSE,
    grade DECIMAL(5,2) NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (enrolled_by) REFERENCES users(id) ON DELETE SET NULL,
    UNIQUE KEY unique_enrollment (course_id, student_id),
    INDEX idx_status (enrollment_status)
);
```

#### 8. `bulk_enrollment_sessions`
```sql
CREATE TABLE bulk_enrollment_sessions (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    course_id BIGINT UNSIGNED NOT NULL,
    uploaded_by BIGINT UNSIGNED NOT NULL,
    file_path VARCHAR(255),
    total_students INT DEFAULT 0,
    successful_enrollments INT DEFAULT 0,
    failed_enrollments INT DEFAULT 0,
    errors JSON,
    status ENUM('processing', 'completed', 'failed') DEFAULT 'processing',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE,
    FOREIGN KEY (uploaded_by) REFERENCES users(id) ON DELETE CASCADE
);
```

---

### المجموعة 2: المحتوى التعليمي (3 جداول)

#### 9. `lessons`
```sql
CREATE TABLE lessons (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    content LONGTEXT,
    objectives TEXT,
    attachments JSON,
    allow_comments BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);
```

#### 10. `videos`
```sql
CREATE TABLE videos (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    video_type ENUM('upload', 'youtube', 'vimeo', 'external') DEFAULT 'upload',
    video_url VARCHAR(500),
    video_path VARCHAR(255),
    thumbnail VARCHAR(255),
    duration INT COMMENT 'seconds',
    quality JSON COMMENT 'e.g. ["720p", "1080p"]',
    subtitles JSON COMMENT 'multiple languages',
    allow_download BOOLEAN DEFAULT FALSE,
    allow_speed_control BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);
```

#### 11. `resources`
```sql
CREATE TABLE resources (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    resource_type ENUM('pdf', 'doc', 'ppt', 'excel', 'image', 'audio', 'other') DEFAULT 'pdf',
    file_path VARCHAR(255) NOT NULL,
    file_name VARCHAR(255) NOT NULL,
    file_size BIGINT COMMENT 'bytes',
    mime_type VARCHAR(100),
    allow_download BOOLEAN DEFAULT TRUE,
    preview_available BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);
```

---

### المجموعة 3: نظام الاختبارات (8 جداول)

#### 12. `question_categories`
```sql
CREATE TABLE question_categories (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    parent_id BIGINT UNSIGNED NULL,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    sort_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (parent_id) REFERENCES question_categories(id) ON DELETE CASCADE
);
```

#### 13. `question_bank`
```sql
CREATE TABLE question_bank (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    category_id BIGINT UNSIGNED,
    created_by BIGINT UNSIGNED,
    question_type ENUM('multiple_choice', 'multiple_response', 'true_false', 'short_answer', 'essay', 'matching', 'ordering', 'fill_in_blank', 'numerical', 'calculated', 'hotspot', 'drag_and_drop') NOT NULL,
    question_text LONGTEXT NOT NULL,
    question_html LONGTEXT,
    question_image VARCHAR(255),
    points DECIMAL(10,2) DEFAULT 1,
    difficulty_level ENUM('easy', 'medium', 'hard') DEFAULT 'medium',
    tags JSON,
    usage_count INT DEFAULT 0,
    is_public BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    deleted_at TIMESTAMP NULL,

    FOREIGN KEY (category_id) REFERENCES question_categories(id) ON DELETE SET NULL,
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_type (question_type),
    INDEX idx_difficulty (difficulty_level)
);
```

#### 14. `quizzes`
```sql
CREATE TABLE quizzes (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    instructions TEXT,
    quiz_type ENUM('practice', 'graded', 'midterm', 'final_exam', 'self_assessment', 'survey') DEFAULT 'graded',
    question_selection_type ENUM('all', 'random', 'manual') DEFAULT 'all',
    questions_per_page INT DEFAULT 0 COMMENT '0 = all in one page',
    passing_score DECIMAL(5,2) DEFAULT 60,
    max_attempts INT NULL COMMENT 'null = unlimited',
    time_limit INT NULL COMMENT 'minutes',
    shuffle_questions BOOLEAN DEFAULT FALSE,
    shuffle_answers BOOLEAN DEFAULT FALSE,
    show_correct_answers ENUM('never', 'after_attempt', 'after_quiz_close') DEFAULT 'after_attempt',
    show_feedback ENUM('never', 'immediately', 'after_attempt') DEFAULT 'after_attempt',
    show_score_immediately BOOLEAN DEFAULT TRUE,
    allow_review BOOLEAN DEFAULT TRUE,
    allow_retake BOOLEAN DEFAULT TRUE,
    retake_wait_time INT COMMENT 'minutes',
    grading_method ENUM('highest', 'average', 'first', 'last') DEFAULT 'highest',
    available_from TIMESTAMP NULL,
    available_until TIMESTAMP NULL,
    created_by BIGINT UNSIGNED,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    deleted_at TIMESTAMP NULL,

    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL
);
```

#### 15. `quiz_questions`
```sql
CREATE TABLE quiz_questions (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    quiz_id BIGINT UNSIGNED NOT NULL,
    question_bank_id BIGINT UNSIGNED NULL,
    question_type ENUM('multiple_choice', 'multiple_response', 'true_false', 'short_answer', 'essay', 'matching', 'ordering', 'fill_in_blank', 'numerical', 'calculated', 'hotspot', 'drag_and_drop') NOT NULL,
    question_text LONGTEXT NOT NULL,
    question_html LONGTEXT,
    question_image VARCHAR(255),
    question_audio VARCHAR(255),
    question_video VARCHAR(255),
    points DECIMAL(10,2) DEFAULT 1,
    difficulty_level ENUM('easy', 'medium', 'hard') DEFAULT 'medium',
    explanation TEXT,
    feedback_correct TEXT,
    feedback_incorrect TEXT,
    case_sensitive BOOLEAN DEFAULT FALSE,
    partial_credit BOOLEAN DEFAULT FALSE,
    auto_grading BOOLEAN DEFAULT TRUE,
    tags JSON,
    sort_order INT DEFAULT 0,
    metadata JSON,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    deleted_at TIMESTAMP NULL,

    FOREIGN KEY (quiz_id) REFERENCES quizzes(id) ON DELETE CASCADE,
    FOREIGN KEY (question_bank_id) REFERENCES question_bank(id) ON DELETE SET NULL
);
```

#### 16. `quiz_question_options`
```sql
CREATE TABLE quiz_question_options (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    question_id BIGINT UNSIGNED NOT NULL,
    option_text TEXT,
    option_html TEXT,
    option_image VARCHAR(255),
    is_correct BOOLEAN DEFAULT FALSE,
    weight DECIMAL(5,2) DEFAULT 0 COMMENT 'for partial credit',
    feedback TEXT,
    sort_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (question_id) REFERENCES quiz_questions(id) ON DELETE CASCADE
);
```

#### 17. `quiz_question_matches`
```sql
CREATE TABLE quiz_question_matches (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    question_id BIGINT UNSIGNED NOT NULL,
    prompt_text TEXT NOT NULL,
    match_text TEXT NOT NULL,
    sort_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (question_id) REFERENCES quiz_questions(id) ON DELETE CASCADE
);
```

#### 18. `quiz_question_blanks`
```sql
CREATE TABLE quiz_question_blanks (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    question_id BIGINT UNSIGNED NOT NULL,
    blank_position INT NOT NULL,
    correct_answers JSON NOT NULL,
    case_sensitive BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (question_id) REFERENCES quiz_questions(id) ON DELETE CASCADE
);
```

#### 19. `question_feedback`
```sql
CREATE TABLE question_feedback (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    question_id BIGINT UNSIGNED NOT NULL,
    student_id BIGINT UNSIGNED NOT NULL,
    attempt_id BIGINT UNSIGNED,
    feedback_type ENUM('report_error', 'suggest_improvement', 'unclear') NOT NULL,
    comment TEXT,
    status ENUM('pending', 'reviewed', 'resolved') DEFAULT 'pending',
    reviewed_by BIGINT UNSIGNED NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (question_id) REFERENCES quiz_questions(id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (reviewed_by) REFERENCES users(id) ON DELETE SET NULL
);
```

---

### المجموعة 4: الاختبارات البرمجية (6 جداول)

#### 20. `programming_questions`
```sql
CREATE TABLE programming_questions (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    question_bank_id BIGINT UNSIGNED NULL,
    title VARCHAR(255) NOT NULL,
    description LONGTEXT NOT NULL,
    difficulty_level ENUM('easy', 'medium', 'hard', 'expert') DEFAULT 'medium',
    programming_language VARCHAR(50) DEFAULT 'python',
    time_limit INT DEFAULT 30 COMMENT 'seconds',
    memory_limit INT DEFAULT 256 COMMENT 'MB',
    allowed_languages JSON,
    starter_code TEXT,
    solution_code TEXT,
    hints JSON,
    topics JSON,
    companies JSON,
    created_by BIGINT UNSIGNED,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    deleted_at TIMESTAMP NULL,

    FOREIGN KEY (question_bank_id) REFERENCES question_bank(id) ON DELETE SET NULL,
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE SET NULL
);
```

#### 21. `test_cases`
```sql
CREATE TABLE test_cases (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    programming_question_id BIGINT UNSIGNED NOT NULL,
    input TEXT,
    expected_output TEXT NOT NULL,
    is_sample BOOLEAN DEFAULT FALSE,
    points DECIMAL(10,2) DEFAULT 1,
    time_limit_override INT NULL COMMENT 'seconds',
    description TEXT,
    sort_order INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (programming_question_id) REFERENCES programming_questions(id) ON DELETE CASCADE
);
```

#### 22. `code_submissions`
```sql
CREATE TABLE code_submissions (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    student_id BIGINT UNSIGNED NOT NULL,
    programming_question_id BIGINT UNSIGNED NOT NULL,
    attempt_id BIGINT UNSIGNED NULL,
    code LONGTEXT NOT NULL,
    language VARCHAR(50) NOT NULL,
    status ENUM('pending', 'running', 'accepted', 'wrong_answer', 'time_limit_exceeded', 'memory_limit_exceeded', 'runtime_error', 'compilation_error') DEFAULT 'pending',
    execution_time INT COMMENT 'milliseconds',
    memory_used INT COMMENT 'KB',
    score DECIMAL(10,2) DEFAULT 0,
    max_score DECIMAL(10,2),
    test_cases_passed INT DEFAULT 0,
    test_cases_total INT DEFAULT 0,
    error_message TEXT,
    output TEXT,
    submitted_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    evaluated_at TIMESTAMP NULL,
    ip_address VARCHAR(45),
    user_agent VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (student_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (programming_question_id) REFERENCES programming_questions(id) ON DELETE CASCADE,
    INDEX idx_status (status),
    INDEX idx_student (student_id)
);
```

#### 23. `submission_test_results`
```sql
CREATE TABLE submission_test_results (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    submission_id BIGINT UNSIGNED NOT NULL,
    test_case_id BIGINT UNSIGNED NOT NULL,
    status ENUM('passed', 'failed', 'error', 'timeout') NOT NULL,
    actual_output TEXT,
    expected_output TEXT,
    execution_time INT COMMENT 'milliseconds',
    memory_used INT COMMENT 'KB',
    error_message TEXT,
    points_earned DECIMAL(10,2) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (submission_id) REFERENCES code_submissions(id) ON DELETE CASCADE,
    FOREIGN KEY (test_case_id) REFERENCES test_cases(id) ON DELETE CASCADE
);
```

#### 24. `code_execution_logs`
```sql
CREATE TABLE code_execution_logs (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    submission_id BIGINT UNSIGNED NOT NULL,
    stdout TEXT,
    stderr TEXT,
    exit_code INT,
    signal VARCHAR(50),
    execution_time INT COMMENT 'milliseconds',
    memory_peak INT COMMENT 'KB',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (submission_id) REFERENCES code_submissions(id) ON DELETE CASCADE
);
```

#### 25. `plagiarism_checks`
```sql
CREATE TABLE plagiarism_checks (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    submission_id BIGINT UNSIGNED NOT NULL,
    compared_with_submission_id BIGINT UNSIGNED NOT NULL,
    similarity_percentage DECIMAL(5,2),
    matching_lines INT,
    algorithm_used ENUM('moss', 'jplag', 'sim', 'custom') DEFAULT 'custom',
    details JSON,
    flagged BOOLEAN DEFAULT FALSE,
    reviewed_by BIGINT UNSIGNED NULL,
    review_notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (submission_id) REFERENCES code_submissions(id) ON DELETE CASCADE,
    FOREIGN KEY (compared_with_submission_id) REFERENCES code_submissions(id) ON DELETE CASCADE,
    FOREIGN KEY (reviewed_by) REFERENCES users(id) ON DELETE SET NULL
);
```

---

### المجموعة 5: الإجابات والتقييم (4 جداول)

#### 26. `student_quiz_attempts`
```sql
CREATE TABLE student_quiz_attempts (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    student_id BIGINT UNSIGNED NOT NULL,
    quiz_id BIGINT UNSIGNED NOT NULL,
    attempt_number INT DEFAULT 1,
    status ENUM('in_progress', 'submitted', 'graded', 'abandoned') DEFAULT 'in_progress',
    started_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    last_activity_at TIMESTAMP NULL,
    submitted_at TIMESTAMP NULL,
    time_spent INT COMMENT 'seconds',
    ip_address VARCHAR(45),
    user_agent VARCHAR(255),
    score DECIMAL(10,2) DEFAULT 0,
    max_score DECIMAL(10,2),
    percentage DECIMAL(5,2),
    passed BOOLEAN DEFAULT FALSE,
    graded_by BIGINT UNSIGNED NULL,
    graded_at TIMESTAMP NULL,
    feedback TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (student_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (quiz_id) REFERENCES quizzes(id) ON DELETE CASCADE,
    FOREIGN KEY (graded_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_student_quiz (student_id, quiz_id)
);
```

#### 27. `student_quiz_answers`
```sql
CREATE TABLE student_quiz_answers (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    attempt_id BIGINT UNSIGNED NOT NULL,
    question_id BIGINT UNSIGNED NOT NULL,
    answer_type ENUM('text', 'json', 'file') DEFAULT 'text',
    answer_text TEXT,
    answer_json JSON,
    answer_file VARCHAR(255),
    selected_options JSON,
    is_correct BOOLEAN NULL,
    points_earned DECIMAL(10,2) DEFAULT 0,
    max_points DECIMAL(10,2),
    auto_graded BOOLEAN DEFAULT TRUE,
    manually_graded BOOLEAN DEFAULT FALSE,
    grader_feedback TEXT,
    time_spent_on_question INT COMMENT 'seconds',
    flagged_for_review BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (attempt_id) REFERENCES student_quiz_attempts(id) ON DELETE CASCADE,
    FOREIGN KEY (question_id) REFERENCES quiz_questions(id) ON DELETE CASCADE
);
```

#### 28. `assignments`
```sql
CREATE TABLE assignments (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    title VARCHAR(255) NOT NULL,
    description LONGTEXT,
    instructions TEXT,
    max_score DECIMAL(10,2) DEFAULT 100,
    submission_type ENUM('text', 'file', 'both') DEFAULT 'both',
    allowed_file_types JSON,
    max_file_size INT COMMENT 'MB',
    max_files INT DEFAULT 5,
    due_date TIMESTAMP NULL,
    late_submission_allowed BOOLEAN DEFAULT FALSE,
    late_penalty_per_day DECIMAL(5,2) COMMENT 'percentage',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);
```

#### 29. `assignment_submissions`
```sql
CREATE TABLE assignment_submissions (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    assignment_id BIGINT UNSIGNED NOT NULL,
    student_id BIGINT UNSIGNED NOT NULL,
    submission_text LONGTEXT,
    files JSON,
    submitted_at TIMESTAMP NULL,
    status ENUM('draft', 'submitted', 'graded', 'returned') DEFAULT 'draft',
    score DECIMAL(10,2) NULL,
    feedback TEXT,
    graded_by BIGINT UNSIGNED NULL,
    graded_at TIMESTAMP NULL,
    is_late BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (assignment_id) REFERENCES assignments(id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (graded_by) REFERENCES users(id) ON DELETE SET NULL,
    UNIQUE KEY unique_submission (assignment_id, student_id)
);
```

---

### المجموعة 6: التتبع والتحليلات (3 جداول)

#### 30. `module_completions`
```sql
CREATE TABLE module_completions (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    student_id BIGINT UNSIGNED NOT NULL,
    module_id BIGINT UNSIGNED NOT NULL,
    completion_status ENUM('not_started', 'in_progress', 'completed') DEFAULT 'not_started',
    score DECIMAL(10,2) NULL,
    time_spent INT COMMENT 'minutes',
    completed_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (student_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (module_id) REFERENCES course_modules(id) ON DELETE CASCADE,
    UNIQUE KEY unique_completion (student_id, module_id)
);
```

#### 31. `student_activity_log`
```sql
CREATE TABLE student_activity_log (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    student_id BIGINT UNSIGNED NOT NULL,
    course_id BIGINT UNSIGNED NULL,
    module_id BIGINT UNSIGNED NULL,
    activity_type ENUM('view', 'start', 'complete', 'download', 'comment', 'submit') NOT NULL,
    activity_data JSON,
    ip_address VARCHAR(45),
    user_agent VARCHAR(255),
    duration INT COMMENT 'seconds',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (student_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE,
    FOREIGN KEY (module_id) REFERENCES course_modules(id) ON DELETE CASCADE,
    INDEX idx_student (student_id),
    INDEX idx_date (created_at)
);
```

#### 32. `student_progress_snapshots`
```sql
CREATE TABLE student_progress_snapshots (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    student_id BIGINT UNSIGNED NOT NULL,
    course_id BIGINT UNSIGNED NOT NULL,
    snapshot_date DATE NOT NULL,
    completion_percentage DECIMAL(5,2),
    modules_completed INT DEFAULT 0,
    total_modules INT DEFAULT 0,
    quiz_average DECIMAL(5,2),
    assignment_average DECIMAL(5,2),
    time_spent INT COMMENT 'minutes',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,

    FOREIGN KEY (student_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE,
    UNIQUE KEY unique_snapshot (student_id, course_id, snapshot_date)
);
```

---

### المجموعة 7: التفاعل (3 جداول)

#### 33. `course_discussions`
```sql
CREATE TABLE course_discussions (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    course_id BIGINT UNSIGNED NOT NULL,
    section_id BIGINT UNSIGNED NULL,
    module_id BIGINT UNSIGNED NULL,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    created_by BIGINT UNSIGNED NOT NULL,
    is_pinned BOOLEAN DEFAULT FALSE,
    is_locked BOOLEAN DEFAULT FALSE,
    views_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    deleted_at TIMESTAMP NULL,

    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE,
    FOREIGN KEY (section_id) REFERENCES course_sections(id) ON DELETE SET NULL,
    FOREIGN KEY (module_id) REFERENCES course_modules(id) ON DELETE SET NULL,
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE CASCADE
);
```

#### 34. `discussion_posts`
```sql
CREATE TABLE discussion_posts (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    discussion_id BIGINT UNSIGNED NOT NULL,
    parent_id BIGINT UNSIGNED NULL,
    user_id BIGINT UNSIGNED NOT NULL,
    content LONGTEXT NOT NULL,
    attachments JSON,
    is_instructor_post BOOLEAN DEFAULT FALSE,
    is_solution BOOLEAN DEFAULT FALSE,
    likes_count INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    deleted_at TIMESTAMP NULL,

    FOREIGN KEY (discussion_id) REFERENCES course_discussions(id) ON DELETE CASCADE,
    FOREIGN KEY (parent_id) REFERENCES discussion_posts(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);
```

#### 35. `course_announcements`
```sql
CREATE TABLE course_announcements (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    course_id BIGINT UNSIGNED NOT NULL,
    created_by BIGINT UNSIGNED NOT NULL,
    title VARCHAR(255) NOT NULL,
    content LONGTEXT NOT NULL,
    priority ENUM('low', 'normal', 'high', 'urgent') DEFAULT 'normal',
    send_email BOOLEAN DEFAULT FALSE,
    is_published BOOLEAN DEFAULT TRUE,
    published_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE,
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE CASCADE
);
```

---

### المجموعة 8: التقييم والشهادات (4 جداول)

#### 36. `grade_items`
```sql
CREATE TABLE grade_items (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    course_id BIGINT UNSIGNED NOT NULL,
    item_type ENUM('quiz', 'assignment', 'programming_challenge', 'participation', 'manual') NOT NULL,
    item_id BIGINT UNSIGNED,
    title VARCHAR(255) NOT NULL,
    max_score DECIMAL(10,2),
    weight DECIMAL(5,2) COMMENT 'percentage',
    include_in_final BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE
);
```

#### 37. `student_grades`
```sql
CREATE TABLE student_grades (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    student_id BIGINT UNSIGNED NOT NULL,
    grade_item_id BIGINT UNSIGNED NOT NULL,
    score DECIMAL(10,2),
    max_score DECIMAL(10,2),
    percentage DECIMAL(5,2),
    letter_grade VARCHAR(5),
    feedback TEXT,
    graded_by BIGINT UNSIGNED,
    graded_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (student_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (grade_item_id) REFERENCES grade_items(id) ON DELETE CASCADE,
    FOREIGN KEY (graded_by) REFERENCES users(id) ON DELETE SET NULL
);
```

#### 38. `certificate_templates`
```sql
CREATE TABLE certificate_templates (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    template_file TEXT COMMENT 'HTML/PDF',
    variables JSON COMMENT 'dynamic variables',
    orientation ENUM('portrait', 'landscape') DEFAULT 'landscape',
    size ENUM('A4', 'Letter') DEFAULT 'A4',
    is_default BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
);
```

#### 39. `course_certificates`
```sql
CREATE TABLE course_certificates (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    course_id BIGINT UNSIGNED NOT NULL,
    student_id BIGINT UNSIGNED NOT NULL,
    template_id BIGINT UNSIGNED,
    certificate_number VARCHAR(100) UNIQUE NOT NULL,
    issue_date DATE NOT NULL,
    expiry_date DATE NULL,
    certificate_data JSON,
    certificate_file VARCHAR(255),
    verification_code VARCHAR(100) UNIQUE NOT NULL,
    verification_url VARCHAR(500),
    is_verified BOOLEAN DEFAULT TRUE,
    digital_signature TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE,
    FOREIGN KEY (student_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (template_id) REFERENCES certificate_templates(id) ON DELETE SET NULL,
    UNIQUE KEY unique_certificate (course_id, student_id)
);
```

#### 40. `course_analytics`
```sql
CREATE TABLE course_analytics (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    course_id BIGINT UNSIGNED NOT NULL,
    date DATE NOT NULL,
    total_enrollments INT DEFAULT 0,
    active_students INT DEFAULT 0,
    completed_students INT DEFAULT 0,
    average_progress DECIMAL(5,2),
    average_score DECIMAL(5,2),
    completion_rate DECIMAL(5,2),
    dropout_rate DECIMAL(5,2),
    engagement_score DECIMAL(5,2),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,

    FOREIGN KEY (course_id) REFERENCES courses(id) ON DELETE CASCADE,
    UNIQUE KEY unique_analytics (course_id, date)
);
```

---

## 🎯 الأنظمة الفرعية

### 1. نظام الاختبارات
- 12 نوع سؤال مختلف
- Auto-grading ذكي
- Manual grading للمقالات
- بنك أسئلة مشترك
- اختبارات عشوائية
- محاولات متعددة
- Proctoring (مراقبة)

### 2. نظام الاختبارات البرمجية
- محرر Monaco Editor
- دعم 10+ لغات برمجة
- Test Cases متعددة
- Execution عبر Judge0/Piston API
- Plagiarism detection
- Leaderboard
- Code templates

### 3. نظام الفواتير والمدفوعات (موجود)
- فواتير تلقائية
- دفع على دفعات
- 7 طرق دفع
- إيصالات قابلة للطباعة

### 4. نظام المعسكرات التدريبية (موجود)
- تسجيل وموافقة
- ربط بالفواتير
- جداول زمنية

---

## 💻 التقنيات المستخدمة

### Backend
- **Framework:** Laravel 11
- **Database:** MySQL 8.0
- **Queue:** Redis
- **Cache:** Redis
- **Storage:** AWS S3 / Local

### Frontend
- **Template Engine:** Blade
- **CSS Framework:** Bootstrap 5
- **JS Framework:** Alpine.js / Vue.js
- **Code Editor:** Monaco Editor
- **WYSIWYG:** TinyMCE
- **Drag & Drop:** SortableJS

### APIs & Services
- **Code Execution:** Judge0 / Piston
- **Video Hosting:** YouTube / Vimeo
- **Email:** AWS SES / Mailgun
- **SMS:** Twilio
- **Payment:** Stripe / PayPal

### DevOps
- **Server:** Ubuntu 22.04
- **Web Server:** Nginx
- **PHP:** 8.2+
- **Process Manager:** Supervisor
- **Deployment:** Docker / Laravel Forge

---

## 📅 خطة التنفيذ (12 أسبوع)

### الأسابيع 1-2: البنية التحتية
- ✅ Migrations للكورسات
- ✅ Models & Relationships
- ✅ Controllers أساسية
- ✅ Routes

### الأسابيع 3-4: المحتوى
- ✅ Lessons, Videos, Resources
- ✅ File uploads
- ✅ Media management

### الأسابيع 5-7: نظام الاختبارات
- ✅ جميع أنواع الأسئلة
- ✅ Auto-grading
- ✅ بنك الأسئلة

### الأسابيع 8-9: الاختبارات البرمجية
- ✅ Monaco Editor
- ✅ Code execution
- ✅ Test cases

### الأسبوع 10: التقييم والشهادات
- ✅ Grades system
- ✅ Certificates

### الأسبوع 11: التفاعل
- ✅ Discussions
- ✅ Announcements

### الأسبوع 12: التحسينات والاختبار
- ✅ Testing
- ✅ Optimization
- ✅ Documentation

---

## 📝 ملاحظات مهمة

1. **الأمان:** جميع الأكواد تنفذ في containers معزولة
2. **الأداء:** استخدام Cache و Queue للعمليات الثقيلة
3. **التوسع:** البنية قابلة للتوسع بسهولة
4. **التوثيق:** كل جدول موثق بالتفصيل
5. **الاختبار:** Unit tests لجميع الوظائف الحرجة

---

## 🔗 روابط مفيدة

- Laravel Docs: https://laravel.com/docs
- Monaco Editor: https://microsoft.github.io/monaco-editor/
- Judge0 API: https://judge0.com
- Piston API: https://github.com/engineer-man/piston

---

**آخر تحديث:** 2025-11-12
**المطور:** أكاديمية كلاودسوفت
**الترخيص:** Proprietary