# 📢 توثيق نظام الإشعارات الكامل

## 📋 نظرة عامة

تم تطوير نظام إشعارات متكامل يجمع بين **Gamification Notifications** (إشعارات داخلية) و **Laravel Notifications** (إيميلات).

---

## 🎯 أنواع الإشعارات المدعومة

### 1️⃣ **إشعارات الشهادات** (جديد ✨)
- **النوع:** `certificate_issued`
- **الوصف:** إشعار عند إصدار شهادة جديدة للطالب
- **القنوات:**
  - ✅ إشعار داخلي (Gamification Notification)
  - ✅ إيميل (Laravel Notification)
- **الأيقونة:** `fas fa-certificate text-success`
- **رابط الإجراء:** صفحة تفاصيل الشهادة

### 2️⃣ **إشعارات الكورسات**
- `course_completed` - إكمال الكورس
- `course_enrolled` - التسجيل في كورس
- `course_reminder` - تذكير بالكورس

### 3️⃣ **إشعارات الاختبارات**
- `quiz_passed` - نجاح في اختبار
- `assignment_graded` - تقييم واجب
- `assignment_due` - موعد تسليم واجب

### 4️⃣ **إشعارات التلعيب**
- `badge_earned` - الحصول على شارة
- `achievement_unlocked` - فتح إنجاز
- `level_up` - الانتقال لمستوى أعلى
- `points_earned` - كسب نقاط
- `points_deducted` - خصم نقاط

### 5️⃣ **إشعارات الفواتير**
- `invoice_created` - إنشاء فاتورة
- `payment_received` - استلام دفعة
- `payment_reminder` - تذكير بالدفع

### 6️⃣ **إشعارات النظام**
- `system_announcement` - إعلان النظام
- `maintenance_notice` - إشعار صيانة

---

## 🗂️ البنية التقنية

### **الجداول المستخدمة:**

#### 1. `gamification_notifications` - الإشعارات الداخلية
```sql
- id
- user_id (FK → users)
- type (string) - نوع الإشعار
- title (string) - العنوان
- message (text) - المحتوى
- icon (string) - الأيقونة
- action_url (string) - رابط الإجراء
- related_type (string) - نوع الكيان المرتبط
- related_id (bigint) - معرف الكيان
- metadata (json) - بيانات إضافية
- is_read (boolean) - هل تم القراءة
- read_at (timestamp) - وقت القراءة
- created_at, updated_at
```

#### 2. `notifications` - إشعارات Laravel (الإيميلات)
```sql
- id (UUID)
- type (string) - اسم كلاس الإشعار
- notifiable_type, notifiable_id
- data (json) - بيانات الإشعار
- read_at (timestamp)
- created_at, updated_at
```

---

## 📦 الملفات الأساسية

### **1. Models:**
- `app/Models/GamificationNotification.php` - نموذج الإشعارات الداخلية
- `app/Models/User.php` - علاقة `gamificationNotifications()`

### **2. Notifications Classes:**
- `app/Notifications/CertificateIssuedNotification.php` - إشعار إيميل الشهادة

### **3. Services:**
- `app/Services/CertificateService.php` - إرسال الإشعارات عند الإصدار

### **4. Controllers:**
- `app/Http/Controllers/Student/Gamification/NotificationController.php` - التحكم بالإشعارات

### **5. Views:**
- `resources/views/student/gamification/notifications/index.blade.php` - صفحة الإشعارات
- `resources/views/student/layouts/main-header.blade.php` - Badge + Dropdown في Header
- `resources/views/student/layouts/main-sidebar.blade.php` - رابط الإشعارات

### **6. Routes:**
```php
Route::prefix('student/gamification/notifications')->group(function() {
    Route::get('/', [NotificationController::class, 'index'])->name('gamification.notifications.index');
    Route::get('/api', [NotificationController::class, 'api'])->name('gamification.notifications.api');
    Route::get('/api/unread-count', [NotificationController::class, 'getUnreadCount']);
    Route::post('/{notification}/mark-as-read', [NotificationController::class, 'markAsRead']);
    Route::post('/mark-all-as-read', [NotificationController::class, 'markAllAsRead']);
    Route::delete('/{notification}', [NotificationController::class, 'destroy']);
});
```

---

## 🔧 كيفية إرسال إشعار شهادة

### **في CertificateService.php:**

```php
use App\Models\GamificationNotification;
use App\Notifications\CertificateIssuedNotification;

// 1. إرسال إشعار Email
$user->notify(new CertificateIssuedNotification($certificate));

// 2. إرسال إشعار داخلي
GamificationNotification::create([
    'user_id' => $user->id,
    'type' => 'certificate_issued',
    'title' => '🎓 تم إصدار شهادتك!',
    'message' => 'تهانينا! تم إصدار شهادة إتمام كورس "' . $course->name . '" بنجاح.',
    'icon' => 'fas fa-certificate text-success fa-lg',
    'action_url' => route('student.certificates.show', $certificate->id),
    'related_type' => Certificate::class,
    'related_id' => $certificate->id,
    'metadata' => [
        'certificate_number' => $certificate->certificate_number,
        'course_name' => $course->name,
        'issue_date' => $certificate->issue_date->format('Y-m-d'),
    ],
    'is_read' => false,
]);
```

---

## 🎨 واجهات المستخدم

### **1. Header Notification Badge:**
- موقع: أعلى يسار الصفحة
- يظهر عدد الإشعارات غير المقروءة
- Dropdown يعرض آخر 10 إشعارات
- تحديث تلقائي كل 3 ثواني
- إشعارات المتصفح (Browser Notifications)
- صوت عند الإشعار الجديد

**المميزات:**
✅ Real-time Updates (كل 3 ثواني)
✅ Browser Notifications
✅ Sound Alert
✅ Badge Counter (99+ للأعداد الكبيرة)
✅ Mark as Read
✅ Mark All as Read
✅ Direct Link to Action

### **2. Sidebar Link:**
- رابط "الإشعارات" في القائمة الجانبية
- Badge يظهر عدد الإشعارات غير المقروءة
- Route: `gamification.notifications.index`

### **3. صفحة الإشعارات الكاملة:**
**المسار:** `/student/gamification/notifications`

**المميزات:**
- ✅ عرض جميع الإشعارات مع Pagination
- ✅ فلتر (الكل / غير المقروءة)
- ✅ فلتر حسب النوع (شهادات، كورسات، اختبارات، إلخ)
- ✅ تحديد الكل كمقروء
- ✅ حذف إشعار واحد
- ✅ ألوان مميزة للإشعارات غير المقروءة
- ✅ أيقونات مخصصة لكل نوع
- ✅ رابط مباشر للإجراء

---

## 📧 إشعارات البريد الإلكتروني

### **CertificateIssuedNotification.php:**

**القنوات المدعومة:**
- `mail` - البريد الإلكتروني
- `database` - قاعدة البيانات (اختياري)

**محتوى الإيميل:**
```
العنوان: 🎓 مبروك! تم إصدار شهادتك
التحية: مرحباً [اسم الطالب]!
المحتوى:
  - تهانينا! 🎉 تم إصدار شهادة إتمام الكورس بنجاح.
  - اسم الكورس: [...]
  - رقم الشهادة: [...]
  - تاريخ الإصدار: [...]
الإجراء: زر "عرض الشهادة" → يذهب لصفحة الشهادة
```

**الإعدادات المطلوبة في `.env`:**
```env
MAIL_MAILER=smtp
MAIL_HOST=smtp.mailtrap.io
MAIL_PORT=2525
MAIL_USERNAME=your_username
MAIL_PASSWORD=your_password
MAIL_ENCRYPTION=tls
MAIL_FROM_ADDRESS="noreply@yourplatform.com"
MAIL_FROM_NAME="${APP_NAME}"
```

---

## 🧪 كيفية الاختبار

### **1. اختبار إشعار الشهادة:**

```bash
# في Tinker
php artisan tinker

# إصدار شهادة تجريبية
$user = User::find(1);
$course = Course::find(1);
$template = CertificateTemplate::first();
$enrollment = $user->enrollments()->where('course_id', $course->id)->first();

$certificateService = app(\App\Services\CertificateService::class);
$certificate = $certificateService->issueCertificate($user, $course, $template, $enrollment);

# التحقق من الإشعارات
$user->gamificationNotifications()->latest()->first();
$user->notifications()->latest()->first();
```

### **2. اختبار Header Badge:**
1. سجل دخول كطالب
2. افتح أي صفحة
3. يجب أن ترى Badge الإشعارات في الـ Header
4. انقر عليه لرؤية Dropdown
5. انقر على إشعار → يجب أن يذهب للصفحة المرتبطة

### **3. اختبار صفحة الإشعارات:**
1. اذهب إلى `/student/gamification/notifications`
2. جرب الفلاتر (الكل / غير المقروءة)
3. جرب فلتر النوع (اختر "الشهادات")
4. انقر "تحديد الكل كمقروء"
5. احذف إشعار واحد

### **4. اختبار Real-time:**
1. افتح صفحة الطالب في متصفحين
2. في المتصفح الأول: أصدر شهادة جديدة (كأدمن)
3. في المتصفح الثاني: يجب أن يظهر الإشعار خلال 3 ثواني
4. يجب أن يظهر Browser Notification
5. يجب سماع صوت الإشعار

---

## 🎯 الملخص

### **ما تم إنجازه:**

✅ **نظام إشعارات متكامل:**
- إشعارات داخلية (Gamification Notifications)
- إشعارات البريد الإلكتروني (Laravel Notifications)

✅ **واجهات مستخدم كاملة:**
- Header Badge + Dropdown
- Sidebar Link
- صفحة الإشعارات الكاملة

✅ **دمج إشعارات الشهادات:**
- إشعار داخلي عند إصدار الشهادة
- إيميل للطالب مع رابط مباشر
- نوع جديد: `certificate_issued`
- فلتر خاص بالشهادات

✅ **مميزات متقدمة:**
- Real-time Updates (كل 3 ثواني)
- Browser Notifications
- Sound Alerts
- Badge Counter
- Mark as Read / Mark All as Read
- Delete Notification
- Pagination
- Type Filters

✅ **التوثيق الكامل:**
- شرح البنية التقنية
- أمثلة كود
- خطوات الاختبار
- Routes + Controllers + Views

---

## 📊 الإحصائيات

| العنصر | الحالة |
|--------|--------|
| **صفحة الإشعارات** | ✅ موجودة وكاملة |
| **Header Badge** | ✅ يعمل + Real-time |
| **Sidebar Link** | ✅ موجود + Badge |
| **إشعارات الشهادات** | ✅ Email + داخلي |
| **فلتر الشهادات** | ✅ مضاف |
| **Real-time Updates** | ✅ كل 3 ثواني |
| **Browser Notifications** | ✅ مفعّل |
| **Sound Alert** | ✅ مفعّل |

---

## 🚀 التطويرات المستقبلية (اختياري)

### **مستوى 1 - التحسينات الأساسية:**
1. ⭐ إعدادات الإشعارات للطالب (تفعيل/تعطيل كل نوع)
2. ⭐ إحصائيات الإشعارات
3. ⭐ تجميع الإشعارات المتشابهة

### **مستوى 2 - التحسينات المتقدمة:**
4. ⭐ Web Push Notifications (خارج المتصفح)
5. ⭐ SMS Notifications
6. ⭐ تخصيص قوالب الإيميلات
7. ⭐ جدولة الإشعارات
8. ⭐ Notification Templates System

---

## 📝 ملاحظات مهمة

### **للأدمن:**
- يمكن إرسال إشعارات مخصصة من لوحة الأدمن
- Route: `admin.notifications.index`
- يدعم إرسال لطالب واحد / مجموعة / كورس / جميع الطلاب

### **للمطور:**
- جميع الإشعارات تستخدم نفس النظام الموحد
- يمكن إضافة أنواع جديدة بسهولة
- Queue مدعوم للإيميلات (ShouldQueue)
- يمكن تخصيص الأيقونات والألوان

### **للطالب:**
- الإشعارات تظهر فوراً في Header
- يمكن فلترة الإشعارات حسب النوع
- يمكن حذف الإشعارات القديمة
- Browser Notifications تحتاج إذن من المتصفح

---

**تاريخ التحديث:** 2025-11-29
**الإصدار:** 1.0
**الحالة:** ✅ مكتمل وجاهز للاستخدام
