# 🔧 استكشاف الأخطاء وحلها - Webhooks System

## المشاكل الشائعة والحلول

### 1. ❌ خطأ: "Table 'w_p_forms_submissions' doesn't exist"

#### الخطأ الكامل:
```
SQLSTATE[42S02]: Base table or view not found: 1146 Table
'claudsoftfinal.w_p_forms_submissions' doesn't exist
(Connection: mysql, SQL: select count(*) as aggregate from `w_p_forms_submissions`)
```

#### السبب:
Laravel يحول اسم الكلاس `WPFormsSubmission` تلقائياً إلى `w_p_forms_submissions` (snake_case) بدلاً من `wpforms_submissions`.

#### الحل:
تم إضافة `protected $table` في Models:

**في WPFormsSubmission.php**:
```php
class WPFormsSubmission extends Model
{
    protected $table = 'wpforms_submissions'; // ✅ تحديد اسم الجدول صراحة

    protected $fillable = [
        // ...
    ];
}
```

**في WebhookLog.php**:
```php
class WebhookLog extends Model
{
    protected $table = 'webhook_logs'; // ✅ تحديد اسم الجدول صراحة

    protected $fillable = [
        // ...
    ];
}
```

#### التحقق من الحل:
```bash
php artisan tinker --execute="echo \App\Models\WPFormsSubmission::count();"
# يجب أن يعمل بدون خطأ
```

---

### 2. ❌ خطأ: "Nothing to migrate" لكن الجداول غير موجودة

#### المشكلة:
```bash
php artisan migrate --path=database/migrations/xxx.php
# INFO  Nothing to migrate.
```
لكن الجداول غير موجودة في قاعدة البيانات.

#### السبب:
الـ Migration تم تسجيله في جدول `migrations` لكن حدث خطأ عند التنفيذ ولم تُنشأ الجداول فعلياً.

#### الحل:
```bash
# الخطوة 1: Rollback
php artisan migrate:rollback --step=1

# الخطوة 2: إعادة التشغيل
php artisan migrate

# الخطوة 3: التحقق
php artisan db:show
```

---

### 3. ❌ خطأ: "Class 'x-webhook.form-fields-display' not found"

#### المشكلة:
عند استخدام Component في Blade:
```blade
<x-webhook.form-fields-display :formData="..." />
```

#### السبب:
الـ Component غير موجود في المسار الصحيح.

#### الحل:
تأكد من وجود الملف في:
```
resources/views/components/webhook/form-fields-display.blade.php
```

الاستخدام الصحيح:
```blade
<x-webhook.form-fields-display
    :formData="$submission->form_data"
    :submissionType="$submission->submission_type"
/>
```

---

### 4. ❌ خطأ: "Route [admin.webhooks.index] not defined"

#### المشكلة:
الروابط في Sidebar لا تعمل.

#### السبب:
Routes غير مُعرّفة أو لم يتم تحميلها.

#### الحل:
تأكد من:

1. **routes/admin.php** يحتوي على:
```php
use App\Http\Controllers\Admin\WebhookManagementController;

Route::prefix('webhooks')->name('admin.webhooks.')->group(function () {
    Route::get('/', [WebhookManagementController::class, 'index'])->name('index');
    Route::get('/submissions', [WebhookManagementController::class, 'submissions'])->name('submissions');
    Route::get('/logs', [WebhookManagementController::class, 'logs'])->name('logs');
    // ...
});
```

2. **routes/admin.php** يتم تحميله في `RouteServiceProvider` أو `bootstrap/app.php`

3. تنظيف الـ cache:
```bash
php artisan route:clear
php artisan route:cache
```

---

### 5. ❌ خطأ: "401 Unauthorized" عند إرسال Webhook

#### المشكلة:
```json
{
    "error": "Unauthorized",
    "message": "Invalid API Key"
}
```

#### السبب:
- API Key خاطئ
- Header غير موجود
- المفتاح في `.env` مختلف عن المرسل

#### الحل:

**في Laravel (.env)**:
```env
WPFORMS_WEBHOOK_SECRET=your-strong-secret-key
```

**في WordPress (Functions.php أو Webhook Settings)**:
```php
'headers' => [
    'X-API-Key' => 'your-strong-secret-key', // نفس المفتاح!
    'Content-Type' => 'application/json'
]
```

**الاختبار**:
```bash
curl -X POST https://yourdomain.com/api/webhooks/wpforms \
  -H "X-API-Key: your-strong-secret-key" \
  -H "Content-Type: application/json" \
  -d '{"form_id":"123","fields":[]}'
```

---

### 6. ❌ خطأ: البيانات لا تُعالج (Status: pending)

#### المشكلة:
الإرسالية تبقى بحالة "pending" ولا تُعالج.

#### الأسباب المحتملة:

1. **الكورس غير موجود**:
   - الحل: أضف الكورس أو استخدم "إعادة المحاولة"

2. **بيانات النموذج غير كاملة**:
   - تحقق من `form_data` في تفاصيل الإرسالية
   - تأكد من وجود حقول: name, email, course

3. **خطأ في الكود**:
   - راجع `storage/logs/laravel.log`
   - راجع `processing_notes` في الإرسالية

#### الحل:
```php
// افتح تفاصيل الإرسالية في Dashboard
// اقرأ "ملاحظات المعالجة"
// أصلح المشكلة
// اضغط "إعادة المحاولة"
```

---

### 7. ❌ خطأ: "Call to undefined method getSubmissionTypes()"

#### المشكلة:
```php
$types = WPFormsSubmission::getSubmissionTypes();
```

#### السبب:
Method غير موجود في Model.

#### الحل:
أضف في `WPFormsSubmission.php`:
```php
public static function getSubmissionTypes(): array
{
    return [
        'enrollment',
        'contact',
        'review',
        'payment',
        'application',
        'survey',
    ];
}
```

---

### 8. ❌ الواجهة لا تظهر في Sidebar

#### المشكلة:
قسم "التكاملات → Webhooks" غير موجود في القائمة الجانبية.

#### السبب:
لم يتم تحديث `main-sidebar.blade.php`.

#### الحل:
تأكد من وجود هذا في `resources/views/admin/layouts/main-sidebar.blade.php`:

```blade
<!-- Start::slide__category -->
<li class="slide__category"><span class="category-name">التكاملات</span></li>
<!-- End::slide__category -->

<!-- Start::slide (Webhooks) -->
<li class="slide has-sub {{ request()->routeIs('admin.webhooks.*') ? 'open active' : '' }}">
    <a href="javascript:void(0);" class="side-menu__item">
        <i class="fas fa-exchange-alt side-menu__icon"></i>
        <span class="side-menu__label">Webhooks</span>
        <i class="fe fe-chevron-right side-menu__angle"></i>
    </a>
    <ul class="slide-menu child1">
        <li><a href="{{ route('admin.webhooks.index') }}">لوحة التحكم</a></li>
        <li><a href="{{ route('admin.webhooks.submissions') }}">الإرساليات</a></li>
        <li><a href="{{ route('admin.webhooks.logs') }}">السجلات</a></li>
    </ul>
</li>
<!-- End::slide -->
```

---

### 9. ❌ Component لا يعرض البيانات بشكل صحيح

#### المشكلة:
البيانات تظهر كـ JSON غير منسق.

#### السبب:
- Component غير مُستخدم
- أو البيانات المُمررة خاطئة

#### الحل:
تأكد من الاستخدام الصحيح:

```blade
{{-- ✅ صحيح --}}
<x-webhook.form-fields-display
    :formData="$submission->form_data"
    :submissionType="$submission->submission_type"
/>

{{-- ❌ خاطئ --}}
<x-webhook.form-fields-display
    formData="{{ $submission->form_data }}"  {{-- لا تستخدم quotes! --}}
/>
```

---

### 10. ❌ خطأ: "CSRF token mismatch"

#### المشكلة:
عند الضغط على "إعادة المحاولة" أو "تنظيف السجلات".

#### السبب:
- Session انتهت
- أو CSRF token غير صحيح

#### الحل:
```blade
{{-- تأكد من وجود @csrf في كل Form --}}
<form action="..." method="POST">
    @csrf
    <button type="submit">إرسال</button>
</form>
```

---

## 🛠️ أوامر التشخيص المفيدة

### التحقق من الجداول:
```bash
php artisan db:show
```

### التحقق من Routes:
```bash
php artisan route:list | grep webhook
```

### التحقق من Models:
```bash
php artisan tinker
>>> \App\Models\WPFormsSubmission::count()
>>> \App\Models\WebhookLog::count()
```

### مشاهدة Logs مباشرة:
```bash
tail -f storage/logs/laravel.log
```

### تنظيف الـ Cache:
```bash
php artisan config:clear
php artisan route:clear
php artisan view:clear
php artisan cache:clear
```

### إعادة تحميل Composer Autoload:
```bash
composer dump-autoload
```

---

## 📞 إذا استمرت المشكلة

1. **راجع Laravel Logs**: `storage/logs/laravel.log`
2. **راجع Webhook Logs**: `/admin/webhooks/logs`
3. **تحقق من قاعدة البيانات**: `php artisan db:show`
4. **اختبر الـ Endpoint**:
```bash
curl https://yourdomain.com/api/webhooks/test
```

---

**آخر تحديث**: 2025-11-28
**الإصدار**: 1.0.0
