# 🔗 دليل ربط WPForms مع Laravel LMS - Webhooks Integration

تم إنشاء نظام متكامل لربط WPForms (WordPress) مع نظام Laravel LMS عبر Webhooks.

---

## ✅ ما تم إنجازه

### 1. قاعدة البيانات (Database Layer)

#### Migration: `2025_11_28_093708_create_wpforms_submissions_table`

**جدول `wpforms_submissions`**:
- تخزين البيانات الواردة من WPForms
- ربط مع المستخدمين والكورسات
- تتبع حالة المعالجة (pending/processed/failed)
- حفظ معلومات IP و User Agent

**جدول `webhook_logs`**:
- تسجيل جميع طلبات الـ Webhooks
- مفيد للتشخيص والتتبع
- حفظ Payload و Headers الكاملة

---

### 2. النماذج (Models)

#### WPFormsSubmission Model
**الموقع**: `app/Models/WPFormsSubmission.php`

**الأنواع المدعومة**:
- `enrollment` - تسجيل في كورس
- `contact` - رسالة تواصل
- `review` - مراجعة كورس
- `payment` - دفع رسوم
- `application` - طلب التحاق
- `survey` - استبيان

**Helper Methods**:
- `markAsProcessed()` - تحديد الحالة كمعالجة
- `markAsFailed()` - تحديد الحالة كفاشلة
- `getFormField()` - استخراج قيمة حقل معين

#### WebhookLog Model
**الموقع**: `app/Models/WebhookLog.php`

يسجل جميع طلبات الـ Webhooks للتتبع والتشخيص.

---

### 3. المتحكم (Controller)

#### WebhookController
**الموقع**: `app/Http/Controllers/Api/WebhookController.php`

**Methods الرئيسية**:

1. **`wpforms(Request $request)`** - معالجة webhook من WPForms
   - تسجيل الطلب في webhook_logs
   - استخراج بيانات النموذج
   - تحديد نوع الإرسال تلقائياً
   - معالجة البيانات حسب النوع

2. **`processEnrollment()`** - معالجة تسجيل الطلاب
   - إنشاء مستخدم جديد أو العثور على موجود
   - تعيين دور "student"
   - البحث عن الكورس
   - تسجيل الطالب في الكورس

3. **`test()`** - نقطة نهاية للاختبار

**الميزات الذكية**:
- كشف تلقائي لنوع النموذج من أسماء الحقول
- استخراج ذكي للحقول بأسماء مختلفة
- معالجة أخطاء شاملة

---

### 4. الأمان (Security)

#### VerifyWebhookSignature Middleware
**الموقع**: `app/Http/Middleware/VerifyWebhookSignature.php`

**طرق المصادقة المدعومة**:

1. **API Key** - مفتاح API في الـ Header
   ```
   X-WPForms-API-Key: your-secret-key
   X-API-Key: your-secret-key
   ```
   أو في Query String:
   ```
   ?api_key=your-secret-key
   ```

2. **Signature Verification** - توقيع HMAC SHA256
   ```
   X-WPForms-Signature: signature-here
   ```

3. **IP Whitelist** - قائمة IPs مسموح بها
   ```php
   'allowed_ips' => ['123.456.789.0']
   ```

4. **Development Mode** - في بيئة local يسمح بدون مصادقة

---

### 5. الإعدادات (Configuration)

#### Webhook Config
**الموقع**: `config/webhooks.php`

```php
'wpforms' => [
    'secret' => env('WPFORMS_WEBHOOK_SECRET'),
    'allowed_ips' => [
        // Add your WordPress server IPs
    ],
    'form_types' => [
        // Map form IDs to types
        // '123' => 'enrollment',
    ],
],
```

#### Environment Variables
في ملف `.env`:
```env
WPFORMS_WEBHOOK_SECRET=your-strong-random-secret-key
```

---

### 6. المسارات (Routes)

**الموقع**: `routes/api.php`

```php
// Test endpoint (no auth)
GET  /api/webhooks/test

// WPForms webhook
POST /api/webhooks/wpforms
```

**URLs الكاملة**:
```
https://yourdomain.com/api/webhooks/test
https://yourdomain.com/api/webhooks/wpforms
```

---

## 🚀 كيفية الإعداد

### الخطوة 1: إعداد Laravel

1. **توليد مفتاح سري**:
```bash
php artisan key:generate --show
```
انسخ المفتاح وضعه في `.env`:
```env
WPFORMS_WEBHOOK_SECRET=base64:xyz123...
```

2. **تشغيل Migrations**:
```bash
php artisan migrate
```

3. **اختبار الـ Endpoint**:
```bash
curl https://yourdomain.com/api/webhooks/test
```

يجب أن ترى:
```json
{
    "success": true,
    "message": "Webhook endpoint is working",
    "timestamp": "2025-11-28T10:30:00+00:00",
    "ip": "123.456.789.0"
}
```

---

### الخطوة 2: إعداد WPForms (WordPress)

#### طريقة 1: استخدام WPForms Webhooks Addon

1. **تثبيت WPForms Webhooks Addon** في WordPress

2. **إعداد Webhook**:
   - اذهب إلى WPForms → Settings → Integrations → Webhooks
   - أضف webhook جديد
   - URL: `https://yourdomain.com/api/webhooks/wpforms`
   - Method: POST
   - Headers:
     ```
     X-API-Key: your-secret-key-from-env
     Content-Type: application/json
     ```

3. **تحديد النموذج**:
   - حدد النموذج المراد ربطه
   - اختر "Send on Form Submit"

4. **تخصيص البيانات المرسلة**:
   ```json
   {
       "form_id": "{form_id}",
       "entry_id": "{entry_id}",
       "fields": [
           {
               "name": "name",
               "value": "{field_id="1"}"
           },
           {
               "name": "email",
               "value": "{field_id="2"}"
           },
           {
               "name": "course",
               "value": "{field_id="3"}"
           }
       ]
   }
   ```

#### طريقة 2: استخدام Code Snippet في WordPress

أضف هذا الكود في `functions.php` أو plugin مخصص:

```php
add_action('wpforms_process_complete', 'send_to_laravel_webhook', 10, 4);

function send_to_laravel_webhook($fields, $entry, $form_data, $entry_id) {
    $webhook_url = 'https://yourdomain.com/api/webhooks/wpforms';
    $api_key = 'your-secret-key-from-env';

    // تحضير البيانات
    $data = [
        'form_id' => $form_data['id'],
        'entry_id' => $entry_id,
        'fields' => []
    ];

    // استخراج الحقول
    foreach ($fields as $field_id => $field) {
        $data['fields'][] = [
            'id' => $field_id,
            'name' => $field['name'] ?? '',
            'value' => $field['value'] ?? '',
            'type' => $field['type'] ?? ''
        ];
    }

    // إرسال البيانات
    $response = wp_remote_post($webhook_url, [
        'headers' => [
            'Content-Type' => 'application/json',
            'X-API-Key' => $api_key
        ],
        'body' => json_encode($data),
        'timeout' => 30
    ]);

    // تسجيل النتيجة (اختياري)
    if (is_wp_error($response)) {
        error_log('WPForms Webhook Error: ' . $response->get_error_message());
    }
}
```

---

### الخطوة 3: إعداد أسماء الحقول في WPForms

للكشف التلقائي عن نوع النموذج، استخدم أسماء حقول واضحة:

**للتسجيل في كورس** (enrollment):
- حقل الاسم: `name`, `full_name`, `student_name`
- حقل البريد: `email`, `student_email`
- حقل الهاتف: `phone`, `mobile`, `telephone`
- حقل الكورس: `course`, `course_name`, `course_title`

**للمراجعات** (review):
- أي حقل يحتوي على `review` أو `rating`

**للدفع** (payment):
- أي حقل يحتوي على `payment` أو `price`

---

## 📋 أمثلة على النماذج

### مثال 1: نموذج تسجيل في كورس

**حقول WPForms**:
1. Name (Text) - `name`
2. Email (Email) - `email`
3. Phone (Phone) - `phone`
4. Course Name (Dropdown) - `course`
   - Options: Laravel, React, Vue, etc.

**النتيجة**:
- يتم إنشاء مستخدم تلقائياً
- تعيين دور student
- تسجيله في الكورس المختار

### مثال 2: نموذج تواصل

**حقول WPForms**:
1. Name (Text)
2. Email (Email)
3. Subject (Text)
4. Message (Textarea)

**النتيجة**:
- يتم حفظ البيانات في wpforms_submissions
- يمكن إرسال إشعار للإدارة

---

## 🔍 التشخيص والمراقبة

### عرض السجلات

```php
// جميع Webhooks المستلمة
$logs = WebhookLog::latest()->get();

// Webhooks الفاشلة
$failed = WebhookLog::failed()->get();

// إرساليات قيد المعالجة
$pending = WPFormsSubmission::pending()->get();

// إرساليات معالجة بنجاح
$processed = WPFormsSubmission::processed()->get();
```

### Laravel Logs

```bash
tail -f storage/logs/laravel.log
```

جميع أخطاء الـ Webhooks يتم تسجيلها مع التفاصيل الكاملة.

---

## 🛠️ استكشاف الأخطاء

### المشكلة: Webhook لا يعمل

**الحلول**:

1. **تحقق من الـ URL**:
   ```bash
   curl https://yourdomain.com/api/webhooks/test
   ```

2. **تحقق من الـ API Key**:
   ```bash
   curl -X POST https://yourdomain.com/api/webhooks/wpforms \
        -H "X-API-Key: your-secret-key" \
        -H "Content-Type: application/json" \
        -d '{"form_id":"123","fields":[]}'
   ```

3. **تحقق من Laravel Logs**:
   ```bash
   tail -f storage/logs/laravel.log
   ```

4. **تحقق من قاعدة البيانات**:
   ```sql
   SELECT * FROM webhook_logs ORDER BY created_at DESC LIMIT 10;
   ```

### المشكلة: 401 Unauthorized

**الأسباب**:
- API Key خاطئ
- لم يتم إضافة Header بشكل صحيح
- Secret في .env مختلف عن المرسل

**الحل**:
تأكد من أن:
```env
WPFORMS_WEBHOOK_SECRET=same-key-in-wordpress
```

### المشكلة: بيانات لا تُعالج

**تحقق من**:
1. نوع النموذج تم كشفه بشكل صحيح
2. أسماء الحقول صحيحة
3. الكورس موجود في قاعدة البيانات
4. سجلات webhook_logs و wpforms_submissions

---

## 🎯 التخصيص المتقدم

### ربط Form ID بنوع معين

في `config/webhooks.php`:
```php
'form_types' => [
    '123' => 'enrollment',    // نموذج التسجيل
    '456' => 'contact',       // نموذج التواصل
    '789' => 'review',        // نموذج المراجعات
],
```

### إضافة معالجة مخصصة

في `WebhookController.php`:

```php
private function processCustomType(WPFormsSubmission $submission): array
{
    $formData = $submission->form_data;

    // معالجتك المخصصة هنا

    $submission->markAsProcessed(null, null, 'Custom processing complete');

    return ['message' => 'Custom submission processed'];
}
```

ثم في `processSubmission()`:
```php
case 'custom_type':
    return $this->processCustomType($submission);
```

### إضافة إشعارات

```php
use Illuminate\Support\Facades\Notification;
use App\Notifications\NewSubmissionNotification;

// في processEnrollment()
Notification::send(
    User::role('admin')->get(),
    new NewSubmissionNotification($submission)
);
```

---

## 📊 Dashboard للإحصائيات

يمكنك إضافة صفحة في لوحة التحكم لعرض:
- عدد الإرساليات اليومية
- معدل النجاح/الفشل
- أكثر النماذج استخداماً
- آخر الإرساليات

```php
// في Admin Controller
public function webhookStats()
{
    $stats = [
        'total' => WPFormsSubmission::count(),
        'today' => WPFormsSubmission::whereDate('created_at', today())->count(),
        'pending' => WPFormsSubmission::pending()->count(),
        'failed' => WPFormsSubmission::failed()->count(),
        'by_type' => WPFormsSubmission::selectRaw('submission_type, count(*) as count')
            ->groupBy('submission_type')
            ->get(),
    ];

    return view('admin.webhooks.stats', compact('stats'));
}
```

---

## 🔐 نصائح الأمان

1. ✅ **استخدم HTTPS دائماً**
2. ✅ **احفظ API Key بشكل آمن** (لا تشاركه في الكود)
3. ✅ **استخدم IP Whitelist** للحماية الإضافية
4. ✅ **راقب السجلات** بانتظام
5. ✅ **قم بتحديث المفاتيح** دورياً
6. ✅ **تحقق من التوقيعات** إذا كان WPForms يدعمها

---

## 📁 ملفات النظام

```
├── app/
│   ├── Http/
│   │   ├── Controllers/
│   │   │   └── Api/
│   │   │       └── WebhookController.php
│   │   └── Middleware/
│   │       └── VerifyWebhookSignature.php
│   └── Models/
│       ├── WPFormsSubmission.php
│       └── WebhookLog.php
│
├── config/
│   └── webhooks.php
│
├── database/
│   └── migrations/
│       └── 2025_11_28_093708_create_wpforms_submissions_table.php
│
├── routes/
│   └── api.php
│
└── bootstrap/
    └── app.php (محدّث)
```

---

## 🎉 الخلاصة

تم بنجاح إنشاء نظام متكامل لربط WPForms مع Laravel LMS عبر Webhooks:

✅ استقبال البيانات من WPForms آمن ومحمي
✅ معالجة تلقائية لتسجيل الطلاب
✅ كشف ذكي لأنواع النماذج
✅ تسجيل شامل للأخطاء والطلبات
✅ مرن وقابل للتوسع

النظام جاهز للإنتاج ويمكن التوسع لدعم أنواع نماذج إضافية! 🚀
