<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Course;
use App\Models\CourseSection;
use App\Models\QuestionBank;
use App\Models\QuestionType;
use App\Models\CourseSectionQuestion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class CourseSectionController extends Controller
{
    /**
     * Display a listing of sections for a course.
     */
    public function index($courseId)
    {
        try {
            $course = Course::with(['sections.modules'])->findOrFail($courseId);
            $sections = $course->sections()->orderBy('sort_order')->get();

            return view('admin.courses.sections.index', compact('course', 'sections'));
        } catch (\Exception $e) {
            return redirect()
                ->route('admin.courses.index')
                ->with('error', 'حدث خطأ أثناء تحميل الأقسام: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for creating a new section.
     */
    public function create($courseId)
    {
        try {
            $course = Course::findOrFail($courseId);
            return view('admin.courses.sections.create', compact('course'));
        } catch (\Exception $e) {
            return redirect()
                ->route('courses.show', $courseId)
                ->with('error', 'حدث خطأ أثناء تحميل نموذج الإنشاء: ' . $e->getMessage());
        }
    }

    /**
     * Store a newly created section.
     */
    public function store(Request $request, $courseId)
    {
        $course = Course::findOrFail($courseId);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'is_visible' => 'nullable|boolean',
            'is_locked' => 'nullable|boolean',
            'show_unavailable' => 'nullable|boolean',
            'available_from' => 'nullable|date',
            'available_until' => 'nullable|date|after:available_from',
        ]);

        DB::beginTransaction();
        try {
            // Get next sort_order
            $maxSortOrder = $course->sections()->max('sort_order') ?? 0;

            $validated['course_id'] = $courseId;
            $validated['sort_order'] = $maxSortOrder + 1;
            $validated['order_index'] = $maxSortOrder + 1; // Set order_index same as sort_order
            $validated['created_by'] = auth()->id();
            $validated['is_visible'] = $request->has('is_visible');
            $validated['is_locked'] = $request->has('is_locked');
            $validated['show_unavailable'] = $request->has('show_unavailable');

            $section = CourseSection::create($validated);

            DB::commit();

            return redirect()
                ->route('courses.show', $courseId)
                ->with('success', 'تم إنشاء القسم بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء إنشاء القسم: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified section.
     */
    public function show($id)
    {
        try {
            $section = CourseSection::with(['course', 'modules.modulable'])->findOrFail($id);
            return view('admin.courses.sections.show', compact('section'));
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->with('error', 'حدث خطأ أثناء تحميل القسم: ' . $e->getMessage());
        }
    }

    /**
     * Show the form for editing the specified section.
     */
    public function edit($id)
    {
        try {
            $section = CourseSection::with('course')->findOrFail($id);
            return view('admin.courses.sections.edit', compact('section'));
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->with('error', 'حدث خطأ أثناء تحميل نموذج التعديل: ' . $e->getMessage());
        }
    }

    /**
     * Update the specified section.
     */
    public function update(Request $request, $id)
    {
        $section = CourseSection::findOrFail($id);

        $validated = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'nullable|string',
            'is_visible' => 'nullable|boolean',
            'is_locked' => 'nullable|boolean',
            'show_unavailable' => 'nullable|boolean',
            'available_from' => 'nullable|date',
            'available_until' => 'nullable|date|after:available_from',
        ]);

        DB::beginTransaction();
        try {
            $validated['updated_by'] = auth()->id();
            $validated['is_visible'] = $request->has('is_visible');
            $validated['is_locked'] = $request->has('is_locked');
            $validated['show_unavailable'] = $request->has('show_unavailable');

            $section->update($validated);

            DB::commit();

            return redirect()
                ->route('courses.show', $section->course_id)
                ->with('success', 'تم تحديث القسم بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'حدث خطأ أثناء تحديث القسم: ' . $e->getMessage());
        }
    }

    /**
     * Remove the specified section.
     */
    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $section = CourseSection::findOrFail($id);
            $courseId = $section->course_id;

            // Check if section has modules
            if ($section->modules()->count() > 0) {
                return redirect()
                    ->back()
                    ->with('error', 'لا يمكن حذف القسم لاحتوائه على محتوى');
            }

            $section->delete();

            DB::commit();

            return redirect()
                ->route('courses.show', $courseId)
                ->with('success', 'تم حذف القسم بنجاح');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()
                ->back()
                ->with('error', 'حدث خطأ أثناء حذف القسم: ' . $e->getMessage());
        }
    }

    /**
     * Reorder sections (drag & drop).
     */
    public function reorder(Request $request, $courseId)
    {
        $validated = $request->validate([
            'sections' => 'required|array',
            'sections.*' => 'exists:course_sections,id',
        ]);

        DB::beginTransaction();
        try {
            foreach ($validated['sections'] as $index => $sectionId) {
                CourseSection::where('id', $sectionId)
                    ->where('course_id', $courseId)
                    ->update(['sort_order' => $index + 1]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'تم إعادة ترتيب الأقسام بنجاح'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء إعادة الترتيب: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Toggle section visibility.
     */
    public function toggleVisibility($id)
    {
        try {
            $section = CourseSection::findOrFail($id);
            $section->is_visible = !$section->is_visible;
            $section->updated_by = auth()->id();
            $section->save();

            $status = $section->is_visible ? 'مرئي' : 'مخفي';

            return redirect()
                ->back()
                ->with('success', "تم تحديث الظهور إلى: {$status}");
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->with('error', 'حدث خطأ أثناء تحديث الظهور: ' . $e->getMessage());
        }
    }

    /**
     * Toggle section lock status.
     */
    public function toggleLock($id)
    {
        try {
            $section = CourseSection::findOrFail($id);
            $section->is_locked = !$section->is_locked;
            $section->updated_by = auth()->id();
            $section->save();

            $status = $section->is_locked ? 'مقفل' : 'مفتوح';

            return redirect()
                ->back()
                ->with('success', "تم تحديث حالة القفل إلى: {$status}");
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->with('error', 'حدث خطأ أثناء تحديث حالة القفل: ' . $e->getMessage());
        }
    }

    // ==================== Question Management ====================

    /**
     * Show questions management page for a section.
     */
    public function manageQuestions($sectionId)
    {
        try {
            $section = CourseSection::with(['course', 'questions.questionType', 'questions.options'])
                ->findOrFail($sectionId);

            // Get available questions from question bank (for import)
            $availableQuestions = QuestionBank::with(['questionType', 'course'])
                ->where(function($query) use ($section) {
                    $query->where('course_id', $section->course_id)
                          ->orWhereNull('course_id');
                })
                ->whereNotIn('id', $section->questions->pluck('id'))
                ->active()
                ->orderBy('created_at', 'desc')
                ->get();

            // Get question types for creating new questions
            $questionTypes = QuestionType::where('is_active', true)
                ->get();

            return view('admin.courses.sections.questions.manage', compact('section', 'availableQuestions', 'questionTypes'));
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->with('error', 'حدث خطأ أثناء تحميل صفحة الأسئلة: ' . $e->getMessage());
        }
    }

    /**
     * Import question from question bank to section.
     */
    public function importQuestion(Request $request, $sectionId)
    {
        $validated = $request->validate([
            'question_id' => 'required|exists:question_bank,id',
            'question_grade' => 'nullable|numeric|min:0',
            'is_required' => 'nullable|boolean',
        ]);

        DB::beginTransaction();
        try {
            $section = CourseSection::findOrFail($sectionId);

            // Check if question already exists in section
            if ($section->questions()->where('question_id', $validated['question_id'])->exists()) {
                return response()->json([
                    'success' => false,
                    'message' => 'هذا السؤال موجود بالفعل في القسم'
                ], 422);
            }

            // Get next order
            $maxOrder = $section->questions()->max('course_section_questions.question_order') ?? 0;

            // Attach question to section
            $section->questions()->attach($validated['question_id'], [
                'question_order' => $maxOrder + 1,
                'question_grade' => $validated['question_grade'] ?? null,
                'is_required' => $validated['is_required'] ?? true,
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'تم استيراد السؤال بنجاح'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء استيراد السؤال: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove question from section.
     */
    public function removeQuestion($sectionId, $questionId)
    {
        DB::beginTransaction();
        try {
            $section = CourseSection::findOrFail($sectionId);
            $section->questions()->detach($questionId);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'تم إزالة السؤال من القسم بنجاح'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء إزالة السؤال: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Reorder section questions.
     */
    public function reorderQuestions(Request $request, $sectionId)
    {
        $validated = $request->validate([
            'questions' => 'required|array',
            'questions.*' => 'exists:question_bank,id',
        ]);

        DB::beginTransaction();
        try {
            $section = CourseSection::findOrFail($sectionId);

            foreach ($validated['questions'] as $index => $questionId) {
                CourseSectionQuestion::where('course_section_id', $sectionId)
                    ->where('question_id', $questionId)
                    ->update(['question_order' => $index + 1]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'تم إعادة ترتيب الأسئلة بنجاح'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء إعادة الترتيب: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update question settings in section.
     */
    public function updateQuestionSettings(Request $request, $sectionId, $questionId)
    {
        $validated = $request->validate([
            'question_grade' => 'nullable|numeric|min:0',
            'is_required' => 'nullable|boolean',
        ]);

        DB::beginTransaction();
        try {
            CourseSectionQuestion::where('course_section_id', $sectionId)
                ->where('question_id', $questionId)
                ->update([
                    'question_grade' => $validated['question_grade'] ?? null,
                    'is_required' => $validated['is_required'] ?? true,
                ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'تم تحديث إعدادات السؤال بنجاح'
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => 'حدث خطأ أثناء التحديث: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show create question form (redirect to question bank with section context).
     */
    public function createQuestion($sectionId, $type)
    {
        try {
            $section = CourseSection::with('course')->findOrFail($sectionId);

            // Store section context in session for later use
            session(['question_creation_context' => [
                'section_id' => $sectionId,
                'course_id' => $section->course_id,
            ]]);

            // Redirect to question bank creation page with type
            return redirect()->route('question-bank.create.type', $type);
        } catch (\Exception $e) {
            return redirect()
                ->back()
                ->with('error', 'حدث خطأ: ' . $e->getMessage());
        }
    }
}
