# 🗄️ Database Schema - تصميم قاعدة البيانات الكامل

<div dir="rtl">

## نظرة عامة

يحتوي نظام Gamification على **14 جدول** جديد تتكامل مع قاعدة البيانات الحالية بدون تعديل أي جداول موجودة.

---

## الجداول (14 Tables)

### 1️⃣ badges (الأوسمة)

**الوصف:** تخزين تعريفات الأوسمة المختلفة التي يمكن للطلاب كسبها.

```sql
CREATE TABLE badges (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL COMMENT 'اسم الوسام',
    slug VARCHAR(255) UNIQUE NOT NULL COMMENT 'معرف فريد',
    description TEXT COMMENT 'وصف الوسام',
    icon VARCHAR(255) COMMENT 'مسار أيقونة الوسام (SVG/PNG)',
    type ENUM('achievement', 'progress', 'special', 'event', 'social')
        DEFAULT 'achievement' COMMENT 'نوع الوسام',
    category VARCHAR(100) COMMENT 'فئة الوسام (quiz, course, engagement...)',
    rarity ENUM('common', 'rare', 'epic', 'legendary', 'mythic')
        DEFAULT 'common' COMMENT 'ندرة الوسام',
    criteria JSON COMMENT 'معايير الحصول على الوسام',
    points_value INT UNSIGNED DEFAULT 0 COMMENT 'النقاط عند الحصول عليه',
    is_active BOOLEAN DEFAULT TRUE COMMENT 'هل الوسام نشط؟',
    is_hidden BOOLEAN DEFAULT FALSE COMMENT 'وسام سري؟',
    order_index INT DEFAULT 0 COMMENT 'ترتيب العرض',
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,

    INDEX idx_type (type),
    INDEX idx_rarity (rarity),
    INDEX idx_active (is_active),
    INDEX idx_category (category)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

**مثال على criteria:**
```json
{
    "type": "quiz_perfect_score",
    "min_count": 5,
    "quiz_type": "any"
}
```

---

### 2️⃣ user_badges (أوسمة الطلاب)

**الوصف:** تخزين الأوسمة التي حصل عليها كل طالب.

```sql
CREATE TABLE user_badges (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id BIGINT UNSIGNED NOT NULL COMMENT 'معرف الطالب',
    badge_id BIGINT UNSIGNED NOT NULL COMMENT 'معرف الوسام',
    earned_at TIMESTAMP NULL COMMENT 'تاريخ الحصول على الوسام',
    progress JSON COMMENT 'تقدم نحو الوسام (إن وجد)',
    is_favorite BOOLEAN DEFAULT FALSE COMMENT 'وسام مفضل للعرض؟',
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,

    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (badge_id) REFERENCES badges(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_badge (user_id, badge_id),
    INDEX idx_user_earned (user_id, earned_at),
    INDEX idx_favorite (user_id, is_favorite)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

---

### 3️⃣ achievements (الإنجازات)

**الوصف:** تعريفات الإنجازات التي يمكن فتحها.

```sql
CREATE TABLE achievements (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL COMMENT 'اسم الإنجاز',
    slug VARCHAR(255) UNIQUE NOT NULL COMMENT 'معرف فريد',
    description TEXT COMMENT 'وصف الإنجاز',
    icon VARCHAR(255) COMMENT 'أيقونة الإنجاز',
    category ENUM('educational', 'social', 'special', 'time_based')
        DEFAULT 'educational' COMMENT 'فئة الإنجاز',
    tier ENUM('bronze', 'silver', 'gold', 'platinum', 'diamond')
        DEFAULT 'bronze' COMMENT 'مستوى الإنجاز',
    criteria JSON NOT NULL COMMENT 'معايير الإنجاز',
    points INT UNSIGNED DEFAULT 0 COMMENT 'النقاط المكتسبة',
    badge_id BIGINT UNSIGNED NULL COMMENT 'وسام مرتبط (اختياري)',
    is_active BOOLEAN DEFAULT TRUE COMMENT 'نشط؟',
    is_hidden BOOLEAN DEFAULT FALSE COMMENT 'إنجاز سري؟',
    order_index INT DEFAULT 0 COMMENT 'ترتيب العرض',
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,

    FOREIGN KEY (badge_id) REFERENCES badges(id) ON DELETE SET NULL,
    INDEX idx_category (category),
    INDEX idx_tier (tier),
    INDEX idx_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

---

### 4️⃣ user_achievements (إنجازات الطلاب)

**الوصف:** تتبع الإنجازات المفتوحة لكل طالب.

```sql
CREATE TABLE user_achievements (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id BIGINT UNSIGNED NOT NULL COMMENT 'معرف الطالب',
    achievement_id BIGINT UNSIGNED NOT NULL COMMENT 'معرف الإنجاز',
    unlocked_at TIMESTAMP NULL COMMENT 'تاريخ فتح الإنجاز',
    progress_data JSON COMMENT 'بيانات التقدم',
    progress_percentage DECIMAL(5,2) DEFAULT 0.00 COMMENT 'نسبة التقدم',
    is_completed BOOLEAN DEFAULT FALSE COMMENT 'مكتمل؟',
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,

    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (achievement_id) REFERENCES achievements(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_achievement (user_id, achievement_id),
    INDEX idx_user_completed (user_id, is_completed),
    INDEX idx_progress (user_id, progress_percentage)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

---

### 5️⃣ points_transactions (سجل معاملات النقاط)

**الوصف:** تسجيل كل عملية منح/خصم نقاط.

```sql
CREATE TABLE points_transactions (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id BIGINT UNSIGNED NOT NULL COMMENT 'معرف الطالب',
    points INT NOT NULL COMMENT 'عدد النقاط (موجب=منح، سالب=خصم)',
    transaction_type ENUM('earned', 'spent', 'bonus', 'penalty', 'refund', 'admin_adjustment')
        DEFAULT 'earned' COMMENT 'نوع المعاملة',
    reason VARCHAR(255) COMMENT 'سبب المعاملة',
    description TEXT COMMENT 'وصف تفصيلي',
    related_type VARCHAR(100) COMMENT 'نوع الكيان المرتبط (polymorphic)',
    related_id BIGINT UNSIGNED COMMENT 'معرف الكيان المرتبط',
    balance_before INT UNSIGNED COMMENT 'الرصيد قبل',
    balance_after INT UNSIGNED COMMENT 'الرصيد بعد',
    metadata JSON COMMENT 'بيانات إضافية',
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,

    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user_date (user_id, created_at),
    INDEX idx_type (transaction_type),
    INDEX idx_related (related_type, related_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

**أمثلة على related_type:**
- `course_completion`
- `quiz_completion`
- `module_completion`
- `daily_login`
- `reward_purchase`

---

### 6️⃣ user_stats (إحصائيات الطلاب)

**الوصف:** إحصائيات شاملة لكل طالب (جدول واحد لكل طالب).

```sql
CREATE TABLE user_stats (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id BIGINT UNSIGNED UNIQUE NOT NULL COMMENT 'معرف الطالب',

    -- النقاط والمستوى
    total_points INT UNSIGNED DEFAULT 0 COMMENT 'إجمالي النقاط',
    available_points INT UNSIGNED DEFAULT 0 COMMENT 'النقاط المتاحة للصرف',
    spent_points INT UNSIGNED DEFAULT 0 COMMENT 'النقاط المصروفة',
    current_level INT UNSIGNED DEFAULT 1 COMMENT 'المستوى الحالي',
    experience_points INT UNSIGNED DEFAULT 0 COMMENT 'نقاط الخبرة (XP)',
    next_level_xp INT UNSIGNED DEFAULT 100 COMMENT 'XP للمستوى القادم',

    -- الإنجازات والأوسمة
    badges_count INT UNSIGNED DEFAULT 0 COMMENT 'عدد الأوسمة',
    achievements_count INT UNSIGNED DEFAULT 0 COMMENT 'عدد الإنجازات',

    -- التقدم التعليمي
    courses_completed INT UNSIGNED DEFAULT 0 COMMENT 'الكورسات المكتملة',
    modules_completed INT UNSIGNED DEFAULT 0 COMMENT 'الوحدات المكتملة',
    quizzes_completed INT UNSIGNED DEFAULT 0 COMMENT 'الاختبارات المكتملة',
    assignments_submitted INT UNSIGNED DEFAULT 0 COMMENT 'الواجبات المسلمة',

    -- الوقت والنشاط
    total_study_time INT UNSIGNED DEFAULT 0 COMMENT 'إجمالي وقت الدراسة (دقائق)',
    total_login_days INT UNSIGNED DEFAULT 0 COMMENT 'إجمالي أيام الدخول',
    current_streak INT UNSIGNED DEFAULT 0 COMMENT 'السلسلة الحالية',
    longest_streak INT UNSIGNED DEFAULT 0 COMMENT 'أطول سلسلة',
    last_activity_at TIMESTAMP NULL COMMENT 'آخر نشاط',

    -- المتصدرين
    global_rank INT UNSIGNED COMMENT 'الترتيب العالمي',
    weekly_rank INT UNSIGNED COMMENT 'الترتيب الأسبوعي',
    monthly_rank INT UNSIGNED COMMENT 'الترتيب الشهري',

    -- بيانات إضافية
    extra_stats JSON COMMENT 'إحصائيات إضافية',

    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,

    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_total_points (total_points),
    INDEX idx_level (current_level),
    INDEX idx_streak (current_streak),
    INDEX idx_rank (global_rank)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

---

### 7️⃣ leaderboards (لوحات المتصدرين)

**الوصف:** تعريفات لوحات المتصدرين المختلفة.

```sql
CREATE TABLE leaderboards (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL COMMENT 'اسم اللوحة',
    slug VARCHAR(255) UNIQUE NOT NULL COMMENT 'معرف فريد',
    description TEXT COMMENT 'وصف اللوحة',
    type ENUM('global', 'course', 'category', 'weekly', 'monthly', 'speed', 'accuracy', 'streak', 'social')
        DEFAULT 'global' COMMENT 'نوع اللوحة',
    period ENUM('all_time', 'yearly', 'monthly', 'weekly', 'daily')
        DEFAULT 'all_time' COMMENT 'الفترة الزمنية',
    criteria JSON COMMENT 'معايير الترتيب',
    max_entries INT UNSIGNED DEFAULT 100 COMMENT 'أقصى عدد للمدخلات',
    is_active BOOLEAN DEFAULT TRUE COMMENT 'نشط؟',
    reset_schedule VARCHAR(50) COMMENT 'جدول إعادة الضبط (cron format)',
    last_reset_at TIMESTAMP NULL COMMENT 'آخر إعادة ضبط',
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,

    INDEX idx_type (type),
    INDEX idx_period (period),
    INDEX idx_active (is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

---

### 8️⃣ leaderboard_entries (مدخلات المتصدرين)

**الوصف:** تخزين الترتيب الفعلي للطلاب في كل لوحة.

```sql
CREATE TABLE leaderboard_entries (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    leaderboard_id BIGINT UNSIGNED NOT NULL COMMENT 'معرف اللوحة',
    user_id BIGINT UNSIGNED NOT NULL COMMENT 'معرف الطالب',
    rank INT UNSIGNED NOT NULL COMMENT 'الترتيب',
    score DECIMAL(10,2) NOT NULL COMMENT 'النقاط/الدرجة',
    division ENUM('bronze', 'silver', 'gold', 'platinum', 'diamond')
        COMMENT 'القسم/الدوري',
    period_start DATE COMMENT 'بداية الفترة',
    period_end DATE COMMENT 'نهاية الفترة',
    metadata JSON COMMENT 'بيانات إضافية',
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,

    FOREIGN KEY (leaderboard_id) REFERENCES leaderboards(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_leaderboard_user_period (leaderboard_id, user_id, period_start),
    INDEX idx_rank (leaderboard_id, rank),
    INDEX idx_division (division),
    INDEX idx_period (period_start, period_end)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

---

### 9️⃣ daily_streaks (سلاسل الحضور اليومي)

**الوصف:** تتبع سلاسل الدخول اليومية للطلاب.

```sql
CREATE TABLE daily_streaks (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id BIGINT UNSIGNED NOT NULL COMMENT 'معرف الطالب',
    streak_date DATE NOT NULL COMMENT 'تاريخ اليوم',
    is_active BOOLEAN DEFAULT TRUE COMMENT 'اليوم ضمن السلسلة النشطة؟',
    login_count INT UNSIGNED DEFAULT 1 COMMENT 'عدد مرات الدخول في هذا اليوم',
    first_login_at TIMESTAMP NULL COMMENT 'أول دخول في اليوم',
    last_login_at TIMESTAMP NULL COMMENT 'آخر دخول في اليوم',
    total_time_minutes INT UNSIGNED DEFAULT 0 COMMENT 'إجمالي الوقت في اليوم (دقائق)',
    activities_completed INT UNSIGNED DEFAULT 0 COMMENT 'الأنشطة المكتملة',
    points_earned INT UNSIGNED DEFAULT 0 COMMENT 'النقاط المكتسبة في اليوم',
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,

    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_date (user_id, streak_date),
    INDEX idx_user_active (user_id, is_active),
    INDEX idx_date (streak_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

---

### 🔟 challenges (التحديات)

**الوصف:** تعريفات التحديات والمسابقات.

```sql
CREATE TABLE challenges (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL COMMENT 'اسم التحدي',
    slug VARCHAR(255) UNIQUE NOT NULL COMMENT 'معرف فريد',
    description TEXT COMMENT 'وصف التحدي',
    type ENUM('daily', 'weekly', 'monthly', 'seasonal', 'special', 'team')
        DEFAULT 'daily' COMMENT 'نوع التحدي',
    difficulty ENUM('easy', 'medium', 'hard', 'expert')
        DEFAULT 'medium' COMMENT 'صعوبة التحدي',
    start_date TIMESTAMP NULL COMMENT 'تاريخ البدء',
    end_date TIMESTAMP NULL COMMENT 'تاريخ الانتهاء',
    criteria JSON NOT NULL COMMENT 'معايير التحدي',
    rewards JSON COMMENT 'المكافآت عند الإكمال',
    max_participants INT UNSIGNED COMMENT 'أقصى عدد للمشاركين',
    is_active BOOLEAN DEFAULT TRUE COMMENT 'نشط؟',
    is_recurring BOOLEAN DEFAULT FALSE COMMENT 'متكرر؟',
    recurrence_pattern VARCHAR(100) COMMENT 'نمط التكرار',
    icon VARCHAR(255) COMMENT 'أيقونة التحدي',
    banner_image VARCHAR(255) COMMENT 'صورة الغلاف',
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,

    INDEX idx_type (type),
    INDEX idx_active (is_active),
    INDEX idx_dates (start_date, end_date),
    INDEX idx_difficulty (difficulty)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

**مثال على criteria:**
```json
{
    "type": "complete_modules",
    "target_count": 5,
    "time_limit_hours": 24
}
```

**مثال على rewards:**
```json
{
    "points": 100,
    "badge_id": 45,
    "reward_item_id": 12
}
```

---

### 1️⃣1️⃣ user_challenges (مشاركة الطلاب في التحديات)

**الوصف:** تتبع مشاركة وتقدم الطلاب في التحديات.

```sql
CREATE TABLE user_challenges (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id BIGINT UNSIGNED NOT NULL COMMENT 'معرف الطالب',
    challenge_id BIGINT UNSIGNED NOT NULL COMMENT 'معرف التحدي',
    status ENUM('not_started', 'in_progress', 'completed', 'failed', 'expired')
        DEFAULT 'not_started' COMMENT 'حالة المشاركة',
    progress_data JSON COMMENT 'بيانات التقدم',
    progress_percentage DECIMAL(5,2) DEFAULT 0.00 COMMENT 'نسبة الإنجاز',
    started_at TIMESTAMP NULL COMMENT 'تاريخ البدء',
    completed_at TIMESTAMP NULL COMMENT 'تاريخ الإكمال',
    rewards_claimed BOOLEAN DEFAULT FALSE COMMENT 'تم استلام المكافآت؟',
    rewards_claimed_at TIMESTAMP NULL COMMENT 'تاريخ استلام المكافآت',
    team_id BIGINT UNSIGNED COMMENT 'معرف الفريق (للتحديات الجماعية)',
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,

    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (challenge_id) REFERENCES challenges(id) ON DELETE CASCADE,
    UNIQUE KEY unique_user_challenge (user_id, challenge_id),
    INDEX idx_status (status),
    INDEX idx_user_status (user_id, status),
    INDEX idx_progress (progress_percentage)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

---

### 1️⃣2️⃣ rewards_catalog (متجر المكافآت)

**الوصف:** كتالوج المكافآت المتاحة للشراء بالنقاط.

```sql
CREATE TABLE rewards_catalog (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    name VARCHAR(255) NOT NULL COMMENT 'اسم المكافأة',
    slug VARCHAR(255) UNIQUE NOT NULL COMMENT 'معرف فريد',
    description TEXT COMMENT 'وصف المكافأة',
    category ENUM('educational', 'digital', 'physical', 'privilege', 'real_world')
        DEFAULT 'educational' COMMENT 'فئة المكافأة',
    type VARCHAR(100) COMMENT 'نوع محدد (discount, course_access, certificate...)',
    cost_points INT UNSIGNED NOT NULL COMMENT 'تكلفة بالنقاط',
    image VARCHAR(255) COMMENT 'صورة المكافأة',
    icon VARCHAR(255) COMMENT 'أيقونة المكافأة',
    stock INT COMMENT 'المخزون المتاح (null = unlimited)',
    availability ENUM('always', 'limited_time', 'seasonal', 'special')
        DEFAULT 'always' COMMENT 'التوفر',
    start_date TIMESTAMP NULL COMMENT 'تاريخ البدء',
    end_date TIMESTAMP NULL COMMENT 'تاريخ الانتهاء',
    requirements JSON COMMENT 'متطلبات الشراء (level, badges...)',
    metadata JSON COMMENT 'بيانات إضافية (discount_percentage, course_id...)',
    is_active BOOLEAN DEFAULT TRUE COMMENT 'نشط؟',
    is_featured BOOLEAN DEFAULT FALSE COMMENT 'مميز؟',
    order_index INT DEFAULT 0 COMMENT 'ترتيب العرض',
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,

    INDEX idx_category (category),
    INDEX idx_cost (cost_points),
    INDEX idx_active (is_active),
    INDEX idx_featured (is_featured)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

---

### 1️⃣3️⃣ user_rewards (المكافآت المستلمة)

**الوصف:** تسجيل المكافآت التي اشتراها الطلاب.

```sql
CREATE TABLE user_rewards (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    user_id BIGINT UNSIGNED NOT NULL COMMENT 'معرف الطالب',
    reward_id BIGINT UNSIGNED NOT NULL COMMENT 'معرف المكافأة',
    claimed_at TIMESTAMP NULL COMMENT 'تاريخ الشراء/الاستلام',
    status ENUM('pending', 'active', 'used', 'expired', 'cancelled')
        DEFAULT 'pending' COMMENT 'الحالة',
    points_spent INT UNSIGNED NOT NULL COMMENT 'النقاط المصروفة',
    expiry_date TIMESTAMP NULL COMMENT 'تاريخ انتهاء الصلاحية',
    used_at TIMESTAMP NULL COMMENT 'تاريخ الاستخدام',
    metadata JSON COMMENT 'بيانات إضافية (tracking codes, vouchers...)',
    notes TEXT COMMENT 'ملاحظات',
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,

    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (reward_id) REFERENCES rewards_catalog(id) ON DELETE CASCADE,
    INDEX idx_user_status (user_id, status),
    INDEX idx_claimed (claimed_at),
    INDEX idx_expiry (expiry_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

---

### 1️⃣4️⃣ experience_levels (تعريف المستويات)

**الوصف:** تعريف مستويات الخبرة والمتطلبات.

```sql
CREATE TABLE experience_levels (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    level INT UNSIGNED UNIQUE NOT NULL COMMENT 'رقم المستوى',
    name VARCHAR(100) COMMENT 'اسم المستوى (Beginner, Expert...)',
    min_xp INT UNSIGNED NOT NULL COMMENT 'أقل XP للمستوى',
    max_xp INT UNSIGNED NOT NULL COMMENT 'أقصى XP للمستوى',
    icon VARCHAR(255) COMMENT 'أيقونة المستوى',
    color VARCHAR(50) COMMENT 'لون المستوى (hex)',
    rewards JSON COMMENT 'مكافآت الوصول للمستوى',
    benefits JSON COMMENT 'مزايا المستوى',
    created_at TIMESTAMP NULL,
    updated_at TIMESTAMP NULL,

    INDEX idx_level (level),
    INDEX idx_xp_range (min_xp, max_xp)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;
```

---

## العلاقات بين الجداول (Relationships)

### User Model
```php
// User.php
public function badges() {
    return $this->belongsToMany(Badge::class, 'user_badges')
        ->withPivot(['earned_at', 'progress', 'is_favorite'])
        ->withTimestamps();
}

public function achievements() {
    return $this->belongsToMany(Achievement::class, 'user_achievements')
        ->withPivot(['unlocked_at', 'progress_data', 'progress_percentage', 'is_completed'])
        ->withTimestamps();
}

public function stats() {
    return $this->hasOne(UserStat::class);
}

public function pointsTransactions() {
    return $this->hasMany(PointsTransaction::class)->latest();
}

public function streaks() {
    return $this->hasMany(DailyStreak::class)->orderBy('streak_date', 'desc');
}

public function challenges() {
    return $this->belongsToMany(Challenge::class, 'user_challenges')
        ->withPivot(['status', 'progress_data', 'progress_percentage', 'started_at', 'completed_at'])
        ->withTimestamps();
}

public function rewards() {
    return $this->belongsToMany(RewardCatalog::class, 'user_rewards')
        ->withPivot(['claimed_at', 'status', 'points_spent', 'expiry_date'])
        ->withTimestamps();
}
```

---

## Indexes للأداء

### Indexes الرئيسية:
- **user_id**: في كل جدول يرتبط بالمستخدم
- **created_at**: لترتيب البيانات زمنياً
- **status/is_active**: للفلترة السريعة
- **composite indexes**: للاستعلامات المتكررة

### أمثلة:
```sql
-- للبحث السريع عن أوسمة طالب معين
INDEX idx_user_earned (user_id, earned_at)

-- لعرض المتصدرين بسرعة
INDEX idx_rank (leaderboard_id, rank)

-- لتتبع السلاسل النشطة
INDEX idx_user_active (user_id, is_active)
```

---

## ملاحظات مهمة

### 1. JSON Fields
استخدام JSON لتخزين بيانات مرنة:
- `criteria`: معايير الأوسمة/الإنجازات/التحديات
- `progress_data`: تقدم الطالب
- `metadata`: بيانات إضافية

### 2. Soft Deletes
لا نستخدم Soft Deletes في هذه الجداول لأن:
- البيانات التاريخية مهمة
- ON DELETE CASCADE يحافظ على نظافة القاعدة

### 3. Timestamps
جميع الجداول تستخدم `created_at` و `updated_at`

### 4. Character Set
جميع الجداول تستخدم `utf8mb4` لدعم العربية والرموز

---

## الخطوة التالية

➡️ بعد مراجعة التصميم، انتقل إلى [01_PHASE_FOUNDATION.md](./01_PHASE_FOUNDATION.md) للبدء بإنشاء الـ Migrations

---

**آخر تحديث:** [تاريخ اليوم]

</div>
